;;; util.scm: Utility functions for uim.
;;;
;;; Copyright (c) 2003,2004 uim Project http://uim.freedesktop.org/
;;;
;;; All rights reserved.
;;;
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above copyright
;;;    notice, this list of conditions and the following disclaimer in the
;;;    documentation and/or other materials provided with the distribution.
;;; 3. Neither the name of authors nor the names of its contributors
;;;    may be used to endorse or promote products derived from this software
;;;    without specific prior written permission.
;;;
;;; THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
;;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
;;; FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
;;; OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;;; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
;;; LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
;;; OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
;;; SUCH DAMAGE.
;;;;

;;
(define control-char?
  (lambda (c)
    (and (number? c)
	 (< c 32))))
;;
(define alphabet-char?
  (lambda (c)
    (and (number? c)
	 (or
	  (and (> c 64) (< c 91))
	  (and (> c 96) (< c 123))))))
;;
(define usual-char?
  (lambda (c)
    (and (number? c)
	 (or
	  (and (> c 32) (< c 127))))))
;;
(define numeral?
  (lambda (c)
    (and (number? c)
	  (and (> c 47) (< c 59)))))

;;
(define to-lower-char
  (lambda (c)
    (if (and (alphabet-char? c) (< c 91))
	(+ c 32)
	c)))
;;
(define string-list-concat
  (lambda (lst)
    (let ((len (length lst)))
      (if (= len 0)
	  ""
	  (string-append
	   (string-list-concat (cdr lst))
	   (car lst))))))
;;
(define string-find
  (lambda (lst str)
    (if lst
	(if
	 (string-equal? (car lst) str)
	 #t
	 (string-find (cdr lst) str))
	())))
;;
(define truncate-list
  (lambda (lst len)
    (if (< (length lst) len)
	nil
	(if (= len 0)
	    '()
	    (cons (car lst)
		  (truncate-list
		   (cdr lst)
		   (- len 1)))))))

;; procedural 'or' for use with 'apply'
;; e.g. (apply proc-or boolean-lst)
;; should be deprecated and replaced with a proper, Schemer's way
(define proc-or
  (lambda xs
    (if (null? xs)
	#f
	(or (car xs)
	    (apply proc-or (cdr xs))))))

;;
;; R5RS procedures (don't expect 100% compatibility)
;;
(define else #t)

(define boolean?
  (lambda (x)
    (or (eq? x #t)
        (eq? x #f))))

(define list?
  (lambda (x)
    (or (null? x)
	(and (pair? x)
	     (list? (cdr x))))))

;; update style-element vars
;; style-spec requires list of (style-element-name . validator)
(define update-style
  (lambda (style-spec style)
    (let* ((elem (car style))
	   (name (car elem))
	   (val (if (symbol? (cdr elem))
		    (symbol-value (cdr elem))
		    (cdr elem)))
	   (spec (assq name style-spec))
	   (valid? (symbol-value (cdr spec))))
      (if (valid? val)
	  (set-symbol-value! name val))
      (if (not (null? (cdr style)))
	  (update-style style-spec (cdr style))))))

;;
;; Preedit color related configurations and functions.
;;
(define reversed-preedit-foreground)
(define reversed-preedit-background)
(define separator-foreground)
(define separator-background)
(define reversed-separator-foreground)
(define reversed-separator-background)

(define uim-color 'uim-color-uim)
(define uim-color-spec
  '((reversed-preedit-foreground   . string?)
    (reversed-preedit-background   . string?)
    (separator-foreground          . string?)
    (separator-background          . string?)
    (reversed-separator-foreground . string?)
    (reversed-separator-background . string?)))

;; predefined color styles
(define uim-color-uim
  '((reversed-preedit-foreground   . "white")
    (reversed-preedit-background   . "black")
    (separator-foreground          . "lightsteelblue")
    (separator-background          . "")
    (reversed-separator-foreground . "white")
    (reversed-separator-background . "black")))
(define uim-color-atok
  '((reversed-preedit-foreground   . "black")
    (reversed-preedit-background   . "blue")
    (separator-foreground          . "lightsteelblue")
    (separator-background          . "")
    (reversed-separator-foreground . "black")
    (reversed-separator-background . "blue")))

;; functions for multiple segments input method
(define multi-segment-type-hiragana 0)
(define multi-segment-type-katakana 1)
(define multi-segment-type-hankana 2)

(define multi-segment-mode-direct 0)
(define multi-segment-mode-hiragana 1)
(define multi-segment-mode-katakana 2)
(define multi-segment-mode-wide-latin 3)
(define multi-segment-mode-hankana 4)

(define multi-segment-make-index-list
  (lambda (n old-lst)
    (if (< n (length old-lst))
	(truncate-list old-lst n 1)
	(append old-lst
		(multi-segment-make-index-list-rec (- n
					(length old-lst)))))))
(define multi-segment-make-index-list-rec
  (lambda (n)
    (if (> n 0)
	(cons 0
	      (multi-segment-make-index-list-rec (- n 1)))
	'())))

(define multi-segment-make-string
  (lambda (sl dir type)
    (let ((get-str-by-type
	   (lambda (l)
	     (cond
	      ((= type multi-segment-type-hiragana)
	       (caar l))
	      ((= type multi-segment-type-katakana)
	       (car (cdar l)))
	      ((= type multi-segment-type-hankana)
	       (cadr (cdar l)))))))
      (if sl
	  (if dir
	      (string-append (multi-segment-make-string (cdr sl) dir type)
			     (get-str-by-type sl))
	      (string-append (get-str-by-type sl)
			     (multi-segment-make-string (cdr sl) dir type)))
	  ""))))

(define multi-segment-make-left-string
  (lambda (sl kana)
    (multi-segment-make-string sl #t kana)))

(define multi-segment-make-right-string
  (lambda (sl kana)
    (multi-segment-make-string sl #f kana)))
