/**/
#include <stdlib.h>
#include <string.h>
#include <m17n.h>
#include "context.h"

static int m17nlib_ok;


static int nr_input_methods;
static struct im_ {
  char *lang;
  char *name;
  MInputMethod *im;
} *im_array;

static int max_input_contexts;
static struct ic_ {
  MInputContext *mic;
} *ic_array;

static int
unused_ic_id()
{
  int i;
  for (i = 0; i < max_input_contexts; i++) {
    if (!ic_array[i].mic) {
      return i;
    }
  }
  ic_array = realloc(ic_array, sizeof(struct ic_) * (max_input_contexts+1));
  ic_array[max_input_contexts].mic = NULL;
  max_input_contexts++;
  return max_input_contexts - 1;
}

static char *
remap_lang_name(char *lang)
{
  static struct lang_map_ {
    char *lib_lang;
    char *lang;
  } lang_map[] = {
    {"Japanese", "ja"},
    {NULL, NULL}
  };
  struct lang_map_ *l;
  for (l = lang_map; l->lib_lang; l++) {
    if (!strcmp(lang, l->lib_lang)) {
      return l->lang;
    }
  }
  return NULL;
}

static void
pushback_input_method(MInputMethod *im,
		      char *lib_lang, char *name)
{
  char *lang = remap_lang_name(lib_lang);
  if (!lang) {
    return ;
  }

  im_array = realloc(im_array, 
		     sizeof(struct im_) * (nr_input_methods + 1));
  im_array[nr_input_methods].lang = strdup(lang);
  im_array[nr_input_methods].name = strdup(name);
  im_array[nr_input_methods].im = im;
  nr_input_methods++;
}

static LISP
init_m17nlib()
{
  MPlist *imlist, *elm;
  M17N_INIT();
  nr_input_methods = 0;
  im_array = NULL;
  max_input_contexts = 0;
  ic_array = NULL;

  imlist = mdatabase_list(msymbol("input-method"), Mnil, Mnil, Mnil);
  for (elm = imlist; mplist_key(elm) != Mnil; elm = mplist_next(elm)) {
    MDatabase *mdb = mplist_value(elm);
    MSymbol *tag = mdatabase_tag(mdb);
    if (tag[1] != Mnil) {
      MInputMethod *im = minput_open_im(tag[1], tag[2], NULL);
      if (im) {
	MSymbol lang = msymbol_get(im->language, Mlanguage);
	pushback_input_method(im, msymbol_name(lang),
			      msymbol_name(im->name));
      }
    }
  }
  m17n_object_unref(imlist);
  m17nlib_ok = 1;
}

void
uim_quit_m17nlib()
{
  if (m17nlib_ok) {
    M17N_FINI();
    m17nlib_ok = 0;
  }
  free(im_array);
  free(ic_array);
}

static LISP
get_nr_input_methods()
{
  return intcons(nr_input_methods);
}

static LISP
get_input_method_name(LISP nth_)
{
  int nth = get_c_long(nth_);
  if (nth < nr_input_methods) {
    char *name = alloca(strlen(im_array[nth].name) + 10);
    sprintf(name, "m17n-%s", im_array[nth].name);
    return rintern(name);
  }
  return NIL;
}

static LISP
get_input_method_lang(LISP nth_)
{
  int nth = get_c_long(nth_);
  if (nth < nr_input_methods) {
    char *lang = im_array[nth].lang;
    return strcons(strlen(lang), lang);
  }
  return NIL;
}

static MInputMethod *
find_im_by_name(char *name)
{
  int i;
  for (i = 0; i < nr_input_methods; i++) {
    if (!strcmp(name, im_array[i].name)) {
      return im_array[i].im;
    }
  }
  return NULL;
}


static LISP
alloc_id(LISP name_)
{
  int id = unused_ic_id();
  char *name = uim_get_c_string(name_);
  MInputMethod *im = find_im_by_name(name);
  ic_array[id].mic = minput_create_ic(im, NULL);
  return intcons(id);
}

static LISP
free_id(LISP id_)
{
  int id = get_c_long(id_);
  if (id < max_input_contexts) {
    if (ic_array[id].mic) {
      minput_destroy_ic(ic_array[id].mic);
      ic_array[id].mic = NULL;
    }
  }
  return NIL;
}

void
uim_init_m17nlib()
{
  init_subr_0("m17nlib-lib-init", init_m17nlib);
  init_subr_0("m17nlib-lib-nr-input-methods", get_nr_input_methods);
  init_subr_1("m17nlib-lib-nth-input-method-lang", get_input_method_lang);
  init_subr_1("m17nlib-lib-nth-input-method-name", get_input_method_name);
  init_subr_1("m17nlib-lib-alloc-context", alloc_id);
  init_subr_1("m17nlib-lib-free-context", free_id);
}
