/*

  Copyright (c) 2003,2004 uim Project http://uim.freedesktop.org/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.

*/

/*
 * gtk+-immodule
 */
#include <gtk/gtk.h>
#include <gtk/gtkimcontext.h>
#include <gtk/gtkimmodule.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <glib/gprintf.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <uim/uim.h>
#include <uim/uim-helper.h>
#include <uim/gettext.h>

#define NR_CANDIDATES 20
#define DEFAULT_SEPARATOR_STR "|"

static int im_uim_fd = -1;
static int read_tag;
static guint snooper_id;

struct preedit_segment {
  int attr;
  char *str;
};

struct candidate_window {
  GtkWidget *top_win;
  GtkWidget *clist;
  GtkWidget *numlabel;
  int nr_candidates;
  int candidate_index;
  int layout_begin;
  int is_active;
  int display_limit;
  int select_count;
  gboolean left;
  GdkRectangle cursor;
};

typedef struct _IMUIMContext {
  /**/
  struct _GtkIMContext parent;
  struct _GtkIMContext *slave;
  uim_context uc;
  struct candidate_window cwin;
  int nr_psegs;
  struct preedit_segment *pseg;
  /**/
  GtkWidget *menu;
  GdkWindow *win;
  GdkWindow *toplevel;
  GdkRectangle preedit_pos; /* preedit_pos not always point the cursor location */
  /**/
  struct _IMUIMContext *prev, *next;
} IMUIMContext;

static IMUIMContext context_list;
static IMUIMContext *focused_context = NULL;
static GObjectClass *parent_class;

typedef struct _IMContextUIMClass 
{
  GtkIMContextClass parent_class;
} IMContextUIMClass;

static void im_uim_class_init (GtkIMContextClass *class);

static void show_preedit(GtkIMContext *ic, GtkWidget *preedit_label);

static void im_uim_helper_disconnect_cb(void);
static void helper_read_cb(gpointer p, int fd, GdkInputCondition c);
static GdkFilterReturn
toplevel_window_candidate_cb (GdkXEvent *xevent, GdkEvent *ev, gpointer data);

static void layout_candidate(IMUIMContext *uic);
static void update_candidate(IMUIMContext *uic);

static void
im_uim_parse_helper_str(char *str);

static gboolean get_user_defined_color(PangoColor *color, const gchar *uim_symbol);

static const GTypeInfo class_info = {
  sizeof(IMContextUIMClass),
  (GBaseInitFunc) NULL,
  (GBaseFinalizeFunc) NULL,
  (GClassInitFunc) im_uim_class_init,
  (GClassFinalizeFunc)NULL,
  NULL,/*for class data*/
  sizeof(IMUIMContext), /* size of instance */
  0,
  (GInstanceInitFunc)NULL, /* constructor */
};

static GType type_im_uim = 0;

#define IM_UIM_CONTEXT(obj) (G_TYPE_CHECK_INSTANCE_CAST((obj),type_im_uim,IMUIMContext))

static const GtkIMContextInfo im_uim_info = {
  "uim",
  "UIM()",
  "gtk+",
  "",
  ""
};

static
void
im_uim_commit_string(void *ptr, char *str)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  g_return_if_fail(str);
  g_signal_emit_by_name(uic, "commit", str);
}

static void
clear_cb(void *ptr)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  int i;
  for (i = 0; i < uic->nr_psegs; i++) {
    free(uic->pseg[i].str);
  }
  free(uic->pseg);
  uic->pseg = NULL;
  uic->nr_psegs = 0;
}

static void
pushback_cb(void *ptr, int attr, char *str)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  g_return_if_fail(str);
  
  if(!strcmp(str,"")
     && !(attr & (UPreeditAttr_Cursor | UPreeditAttr_Separator)))
    return;

  uic->pseg = realloc(uic->pseg,
		      sizeof(struct preedit_segment) *
		      (uic->nr_psegs + 1));
  uic->pseg[uic->nr_psegs].str = g_strdup(str);
  uic->pseg[uic->nr_psegs].attr = attr;
  uic->nr_psegs ++;
}

static void
update_cb(void *ptr)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  g_return_if_fail(uic);
  g_signal_emit_by_name(uic, "preedit_changed");
}

static int
convert_keyval(int key)
{
  switch (key) {
  case GDK_BackSpace: return UKey_Backspace;
  case GDK_Delete: return UKey_Delete;
  case GDK_Escape: return UKey_Escape;
  case GDK_Tab: return UKey_Tab;
  case GDK_Return: return UKey_Return;
  case GDK_Left: return UKey_Left;
  case GDK_Up: return UKey_Up;
  case GDK_Right: return UKey_Right;
  case GDK_Down: return UKey_Down;
  case GDK_Prior: return UKey_Prior;
  case GDK_Next: return UKey_Next;
  case GDK_Home: return UKey_Home;
  case GDK_End: return UKey_End;
  case GDK_Zenkaku_Hankaku: return UKey_Zenkaku_Hankaku;
  case GDK_Multi_key: return UKey_Multi_key;
  case GDK_Mode_switch: return UKey_Mode_switch;
  case GDK_Henkan_Mode: return UKey_Henkan_Mode;
  case GDK_Muhenkan: return UKey_Muhenkan;
  case GDK_Shift_L: return UKey_Shift_key;
  case GDK_Shift_R: return UKey_Shift_key;
  case GDK_Control_L: return UKey_Control_key;
  case GDK_Control_R: return UKey_Control_key;
  case GDK_Alt_L: return UKey_Alt_key;
  case GDK_Alt_R: return UKey_Alt_key;
  case GDK_Meta_L: return UKey_Meta_key;
  case GDK_Meta_R: return UKey_Meta_key;
  case GDK_Super_L: return UKey_Super_key;
  case GDK_Super_R: return UKey_Super_key;
  case GDK_Hyper_L: return UKey_Hyper_key;
  case GDK_Hyper_R: return UKey_Hyper_key;
  }

  if(key >= GDK_F1 && key <= GDK_F35) {
    return key - GDK_F1 + UKey_F1;
  }
  if(key >= GDK_KP_0 && key <= GDK_KP_9) {
    return key - GDK_KP_0 + UKey_0;
  }
  if (key < 256) {
    return key;
  }
  return UKey_Other;
}

static int
convert_modifier(int mod)
{
  int rv = 0;
  if (mod & GDK_SHIFT_MASK) {
    rv |= UMod_Shift;
  }
  if (mod & GDK_CONTROL_MASK) {
    rv |= UMod_Control;
  }
  if (mod & GDK_MOD1_MASK) {
    rv |= UMod_Alt;
  }
  if (mod & GDK_MOD3_MASK) {  /* assuming mod3 */
    rv |= UMod_Super;
  }
  if (mod & GDK_MOD4_MASK) {  /* assuming mod4 */
    rv |= UMod_Hyper;
  }
  return rv;
}


/*
 * KEY EVENT HANDLER
 */
static gboolean
filter_keypress(GtkIMContext *ic,
		GdkEventKey *key)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  return gtk_im_context_filter_keypress(uic->slave, key);
}

static gboolean
get_user_defined_color(PangoColor *color, const gchar *uim_symbol)
{
  gboolean parsed = FALSE;
  gchar *literal = uim_symbol_value_str(uim_symbol);
  if(literal != NULL && literal[0] != '\0') {
    parsed = pango_color_parse(color, literal);
  }
  g_free(literal);
  return parsed;
}

static char *
get_preedit_segment(struct preedit_segment *ps,
		    PangoAttrList *attrs,
		    char *str)
{
  PangoAttribute *attr;
  const gchar *segment_str = ps->str;
  if ((ps->attr & UPreeditAttr_Separator) && !strcmp(segment_str, "")) {
    segment_str = DEFAULT_SEPARATOR_STR;
  }
  if (attrs) {
    PangoColor color;
    int begin, end;
    begin = strlen(str);
    end = begin + strlen(segment_str);
    if (ps->attr & UPreeditAttr_UnderLine) {
      attr = pango_attr_underline_new(PANGO_UNDERLINE_SINGLE);
      attr->start_index = begin;
      attr->end_index = end;
      pango_attr_list_change(attrs, attr);
    }
    if (ps->attr & UPreeditAttr_Separator) {
      const gchar *separator_fg_symbol, *separator_bg_symbol;
      if (ps->attr & UPreeditAttr_Reverse) {
	separator_fg_symbol = "reversed-separator-foreground";
	separator_bg_symbol = "reversed-separator-background";
      } else {
	separator_fg_symbol = "separator-foreground";
	separator_bg_symbol = "separator-background";
      }
      if (get_user_defined_color(&color, separator_fg_symbol))
      {
	attr = pango_attr_foreground_new (color.red, color.green, color.blue);
	attr->start_index = begin;
	attr->end_index = end;
	pango_attr_list_change (attrs, attr);
      }

      if (get_user_defined_color(&color, separator_bg_symbol))
      {
	attr = pango_attr_background_new (color.red, color.green, color.blue);
	attr->start_index = begin;
	attr->end_index = end;
	pango_attr_list_change (attrs, attr);
      }
    } else if (ps->attr & UPreeditAttr_Reverse) {
      if (get_user_defined_color(&color, "reversed-preedit-foreground")
	  || pango_color_parse(&color, "#fff"))
      {
	attr = pango_attr_foreground_new (color.red, color.green, color.blue);
	attr->start_index = begin;
	attr->end_index = end;
	pango_attr_list_change (attrs, attr);
      }

      if (get_user_defined_color(&color, "reversed-preedit-background")
	  || pango_color_parse(&color, "#000"))
      {
	attr = pango_attr_background_new (color.red, color.green, color.blue);
	attr->start_index = begin;
	attr->end_index = end;
	pango_attr_list_change (attrs, attr);
      }
    }
  }
  str = (char *)realloc(str, strlen(str) + strlen(segment_str)+1);
  strcat(str, segment_str);
  return str;
}


static void
im_uim_get_preedit_string(GtkIMContext *ic, gchar **str,
			  PangoAttrList **attrs,
			  gint *cursor_pos)
{
  char *tmp;
  int i, pos = 0;
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);

  if (attrs) {
    *attrs = pango_attr_list_new();
  }
  tmp = g_strdup("");
  for (i = 0; i < uic->nr_psegs; i++) {
    if (uic->pseg[i].attr & UPreeditAttr_Cursor) {
      pos = g_utf8_strlen(tmp,-1);
    }
    if (attrs) {
      tmp = get_preedit_segment(&uic->pseg[i], *attrs, tmp);
    } else {
      tmp = get_preedit_segment(&uic->pseg[i], NULL, tmp);
    }
  }
  if (cursor_pos) {
    *cursor_pos = pos;
  }
  if (str) {
    *str = tmp;
  } else {
    free(tmp);
  }
}

static void
im_uim_set_cursor_location(GtkIMContext *ic,
			   GdkRectangle *area)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);

  if(uic->cwin.left &&  uic->nr_psegs > 1){
    return;
  }

  uic->cwin.cursor.x= area->x;
  uic->cwin.cursor.y= area->y;
  uic->cwin.cursor.height= area->height;
  uic->cwin.cursor.width= area->width;

  uic->preedit_pos.x = area->x;
  uic->preedit_pos.y = area->y;
}


static void
im_uim_commit_cb(GtkIMContext *ic,
		 const gchar *str,
		 IMUIMContext *is)
{
  g_return_if_fail(str);
  g_signal_emit_by_name(is, "commit", str);
}

static void
check_helper_connection()
{
  if(im_uim_fd < 0) {
    im_uim_fd     = uim_helper_init_client_fd(im_uim_helper_disconnect_cb);
    if (im_uim_fd >= 0) {
      read_tag = gdk_input_add(im_uim_fd, (GdkInputCondition)GDK_INPUT_READ,
			       helper_read_cb, 0);
    }
  }
}

static void
focus_in(GtkIMContext *ic)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  IMUIMContext *cc;

  focused_context = uic;

  check_helper_connection();

  uim_prop_list_update(uic->uc);
  uim_prop_label_update(uic->uc);

  for (cc = context_list.next; cc != &context_list; cc = cc->next) {
    if (cc != uic) {
      if (cc->cwin.top_win) {
	gtk_widget_hide(cc->cwin.top_win);
      }
    }
  }
  
  if(uic->cwin.top_win && uic->cwin.is_active){
    gtk_widget_show(uic->cwin.top_win);
  }
  
  uim_helper_client_focus_in(uic->uc);
}

static void
focus_out(GtkIMContext *ic)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);

  if(focused_context == uic)
    focused_context = NULL;

  uim_helper_client_focus_out(uic->uc);

  if(uic->cwin.top_win){
    gtk_widget_hide(uic->cwin.top_win);
  }
}

void im_uim_reset(GtkIMContext *ic)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);

  uim_reset_context(uic->uc);

  //  g_warning (_("reset called: implement me"));

}

void
set_use_preedit(GtkIMContext *ic, gboolean use_preedit)
{
  GtkWidget *preedit_window;
  GtkWidget *preedit_label;
  if( use_preedit == FALSE ) {
    preedit_window = gtk_window_new(GTK_WINDOW_POPUP);
    preedit_label = gtk_label_new("");
    gtk_container_add(GTK_CONTAINER(preedit_window), preedit_label);
    
    g_signal_connect(G_OBJECT(ic), "preedit-changed",
		     G_CALLBACK(show_preedit), preedit_label );
    gtk_widget_show_all(preedit_window);
  }
}


static void
show_preedit(GtkIMContext *ic, GtkWidget *preedit_label)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  GtkWidget *preedit_window;
  gchar *str;
  gint cursor_pos;
  PangoAttrList *attrs;

  preedit_window = gtk_widget_get_parent(preedit_label);
  
  gtk_im_context_get_preedit_string (ic, &str, &attrs, &cursor_pos);

  
  if ( strlen (str) > 0) {
      gint x,y,w,h;
      PangoLayout *layout;

      gtk_label_set_text(GTK_LABEL(preedit_label), str);
      gtk_label_set_attributes(GTK_LABEL(preedit_label), attrs);

      gdk_window_get_origin(uic->win, &x, &y);
      
      gtk_window_move( GTK_WINDOW(preedit_window),
		       x + uic->preedit_pos.x,
		       y + uic->preedit_pos.y );

      layout = gtk_label_get_layout(GTK_LABEL(preedit_label));

      pango_layout_get_cursor_pos (layout, 0, NULL, NULL);

      pango_layout_get_pixel_size (layout, &w, &h);
      gtk_window_resize (GTK_WINDOW(preedit_window), w, h);

      gtk_widget_show( preedit_window );

    }
  else {
      gtk_label_set_text(GTK_LABEL(preedit_label), "");
      gtk_widget_hide( preedit_window );
      gtk_window_resize( GTK_WINDOW(preedit_window), 0, 0 );
  }
  g_free (str);
  pango_attr_list_unref (attrs);
  
}

static GdkFilterReturn
toplevel_window_candidate_cb (GdkXEvent *xevent, GdkEvent *ev, gpointer data)
{
  IMUIMContext *uic = data;

  if(!uic)
    return GDK_FILTER_CONTINUE;

  if(uic->cwin.top_win && uic->cwin.is_active ){
    layout_candidate(uic);
  }
  return GDK_FILTER_CONTINUE;
}


static void
set_client_window(GtkIMContext *ic, GdkWindow *w)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);

  if(w) {
    g_object_ref(w);
  }
  uic->win = w;
}


/*
 * DESTRUCTOR
 */
static void
im_uim_finalize(GObject *obj)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(obj);
  //  set_client_window(GTK_IM_CONTEXT(uic), NULL);

  uic->next->prev = uic->prev;
  uic->prev->next = uic->next;

  if (uic->menu) {
    gtk_widget_destroy(uic->menu);
  }
  if (uic->cwin.top_win) {
    gtk_widget_destroy(uic->cwin.top_win);
  }
  uic->menu = NULL;

  uim_release_context(uic->uc);
  
  g_signal_handlers_disconnect_by_func(uic->slave,
				       im_uim_commit_cb,
				       uic);
  g_object_unref(uic->slave);
  parent_class->finalize (obj);

  if(uic == focused_context)
    focused_context = NULL;
}

static void
im_uim_class_init (GtkIMContextClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  parent_class = g_type_class_peek_parent(class);
  class->set_client_window = set_client_window;
  class->filter_keypress = filter_keypress;
  class->get_preedit_string = im_uim_get_preedit_string;
  class->set_cursor_location = im_uim_set_cursor_location;
  class->focus_in = focus_in;
  class->focus_out = focus_out;
  class->reset = im_uim_reset;
  class->set_use_preedit = set_use_preedit;

  object_class->finalize = im_uim_finalize;
}

static void
update_prop_list_cb(void *ptr, char *str)
{
  GString *tmp = g_string_new("");
  g_string_printf(tmp, "prop_list_update\ncharset=UTF-8\n%s", str);

  uim_helper_send_message(im_uim_fd, tmp->str);
  g_string_free(tmp, TRUE);
}

static void
update_prop_label_cb(void *ptr, char *str)
{
  GString *tmp = g_string_new("");
  g_string_printf(tmp, "prop_label_update\ncharset=UTF-8\n%s", str);

  uim_helper_send_message(im_uim_fd, tmp->str);
  g_string_free(tmp, TRUE);
}


/* This function is called by Gtk+, when user select candidate
   by clicking candidate, etc. */
static void
select_candidate_cb(GtkWidget *clist,
		    gint row, gint column,
		    GdkEventButton *event,
		    gpointer data)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(data);

  if (uic->cwin.select_count == 0) {
    int idx = row + uic->cwin.layout_begin;
  
    if( idx >= uic->cwin.nr_candidates ) {
      gtk_clist_select_row(GTK_CLIST(uic->cwin.clist),
			   uic->cwin.candidate_index - uic->cwin.layout_begin,
			   0);
      return;
    }

    uim_set_candidate_index(uic->uc, idx);
    uic->cwin.candidate_index = idx;
  } else {
    uic->cwin.select_count = 0;
  }
}

static void
init_candidate_win(IMUIMContext *uic)
{
  struct candidate_window *cwin = &uic->cwin;
  GtkWidget *vbox;
  GtkWidget *numlabel;
  gchar *leftp;

  if (cwin->top_win) {
    return ;
  }

  leftp = uim_symbol_value_str("candidate-window-position");
 
  if(leftp && !strcmp(leftp,"left")) {
    cwin->left = TRUE;
  } else {
    cwin->left = FALSE;
  }
  g_free(leftp);

  cwin->top_win = gtk_window_new(GTK_WINDOW_POPUP);
  cwin->clist = gtk_clist_new(2);
  cwin->is_active = 0;
  g_signal_connect(G_OBJECT(cwin->clist), "select_row",
		   G_CALLBACK(select_candidate_cb),
		   uic);
  vbox = gtk_vbox_new(FALSE, 0);
  numlabel = gtk_label_new("");
  cwin->numlabel = numlabel;

  gtk_box_pack_start(GTK_BOX(vbox), cwin->clist, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), numlabel, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(cwin->top_win), vbox);
  {
    gint x,y,w,h,d;
    gdk_window_get_geometry(uic->win, &x, &y,&w,&h,&d);
    gdk_window_get_origin(uic->win, &x, &y);
    gtk_window_move(GTK_WINDOW(cwin->top_win), x , y+h);
  }

  gtk_clist_set_column_auto_resize( GTK_CLIST(cwin->clist), 0, TRUE );
  gtk_clist_set_column_auto_resize( GTK_CLIST(cwin->clist), 1, TRUE );

  gtk_clist_set_shadow_type( GTK_CLIST(cwin->clist), GTK_SHADOW_NONE );
  
  gtk_widget_show(cwin->clist);
  gtk_widget_show(numlabel);
  gtk_widget_show(vbox);
  
}

static int
trim_index(IMUIMContext *uic)
{
  struct candidate_window *cwin = &uic->cwin;
  int changed = 0;

  if(cwin->display_limit != 0){
    
    while (cwin->layout_begin + cwin->display_limit <= cwin->candidate_index) {
      changed = 1;
      cwin->layout_begin += cwin->display_limit;
    }  
    while (cwin->layout_begin > cwin->candidate_index) {
      changed = 1;
      cwin->layout_begin -= cwin->display_limit;
    }
    if (cwin->layout_begin < 0) {
      changed = 1;
      cwin->layout_begin = 0;
    }
  }
  else {
    while (cwin->layout_begin + NR_CANDIDATES <= cwin->candidate_index) {
      changed = 1;
      cwin->layout_begin += NR_CANDIDATES;
    }  
    while (cwin->layout_begin > cwin->candidate_index) {
      changed = 1;
      cwin->layout_begin -= NR_CANDIDATES;     
    }
    if (cwin->layout_begin < 0) {
      changed = 1;
      cwin->layout_begin = 0;
    }
  }
  return changed;
}


static void
calc_candidate_window_size(struct candidate_window cwin, int *width, int *height)
{
  GtkRequisition req_clist;
  GtkRequisition req_numlabel;
  gtk_widget_size_request(GTK_WIDGET(cwin.clist), &req_clist);
  gtk_widget_size_request(GTK_WIDGET(cwin.numlabel), &req_numlabel);

  *height = req_clist.height + req_numlabel.height;
  *width  = req_clist.width;

  if(*width < 80)
    *width = 80;

  gtk_window_resize(GTK_WINDOW(cwin.top_win), *width, *height);
}

static void
layout_candidate(IMUIMContext *uic)
{
  int  i, x, y, topwin_x, topwin_y;
  int  sc_he, cw_he; /*screen height, candidate window height*/
  int  sc_wi, cw_wi;
  char idx[20];
  char *old_cand_str = NULL;
  char *new_cand_str = NULL;

  trim_index(uic);

  calc_candidate_window_size(uic->cwin, &cw_wi, &cw_he);

  sc_he = gdk_screen_get_height(gdk_screen_get_default ());
  sc_wi = gdk_screen_get_width (gdk_screen_get_default ());

  gdk_window_get_origin(uic->win, &topwin_x, &topwin_y);

  if(sc_wi <  topwin_x + uic->cwin.cursor.x + cw_wi ) {
    x = topwin_x + uic->cwin.cursor.x - cw_wi;
  } else {
    x = topwin_x + uic->cwin.cursor.x;
  }

  if(sc_he <  topwin_y + uic->cwin.cursor.y +  uic->cwin.cursor.height + cw_he ) {
    y = topwin_y + uic->cwin.cursor.y - cw_he;
  } else {
    y = topwin_y + uic->cwin.cursor.y +  uic->cwin.cursor.height;
  }

  gtk_window_move(GTK_WINDOW(uic->cwin.top_win), x, y );
  
  /* XXX: Consider if display_limit == 0 */
  for (i = 0; i < NR_CANDIDATES && i < uic->cwin.display_limit; i++) {
    if (i < uic->cwin.nr_candidates - uic->cwin.layout_begin) {
      uim_candidate new_cand = uim_get_candidate(uic->uc, uic->cwin.layout_begin + i, i);
      g_return_if_fail(new_cand);
      
      new_cand_str = (gchar*)uim_candidate_get_cand_str(new_cand);
      g_return_if_fail(new_cand_str);

      gtk_clist_get_text(GTK_CLIST(uic->cwin.clist), i, 1, &old_cand_str);

      /* Checking old cand and new cand is the same or not. If not, call update. 
	 Caching new result is good idea for speed up and should be implement.*/
      if(!old_cand_str || strcmp(new_cand_str, old_cand_str) != 0) {
	update_candidate(uic);
	uim_candidate_free(new_cand);
	break;
      }
      uim_candidate_free(new_cand);
    }
  }
  
  uic->cwin.select_count = 1;

  if(uic->cwin.candidate_index >= 0) {
    gtk_clist_select_row(GTK_CLIST(uic->cwin.clist),
			 uic->cwin.candidate_index - uic->cwin.layout_begin,
			 0);
    
    sprintf(idx, "%d / %d", uic->cwin.candidate_index + 1 , uic->cwin.nr_candidates); 
  } else {
    gtk_clist_unselect_all(GTK_CLIST(uic->cwin.clist));
    sprintf(idx, "- / %d", uic->cwin.nr_candidates); 
  }
  gtk_label_set_text( GTK_LABEL(uic->cwin.numlabel), idx);
  
}


static void
update_candidate(IMUIMContext *uic)
{
  char *buf[2];
  int i;
  uim_candidate cand = NULL;
  gtk_clist_freeze(GTK_CLIST(uic->cwin.clist));
  gtk_clist_clear(GTK_CLIST(uic->cwin.clist));
  
  /* XXX: Consider if display_limit == 0 */
  for (i = 0; i < NR_CANDIDATES && i < uic->cwin.display_limit; i++) {
    if (i < uic->cwin.nr_candidates - uic->cwin.layout_begin) {
      cand = uim_get_candidate(uic->uc, uic->cwin.layout_begin + i, i);
      buf[0] = (char *) uim_candidate_get_heading_label(cand);
      buf[1] = (char *) uim_candidate_get_cand_str(cand);
      gtk_clist_append(GTK_CLIST(uic->cwin.clist), buf);
      uim_candidate_free(cand);
    } else {
      buf[0] = "";
      buf[1] = "";
      gtk_clist_append(GTK_CLIST(uic->cwin.clist), buf);
    }
  }
  gtk_clist_thaw( GTK_CLIST(uic->cwin.clist) );
}

static void
cand_activate_cb(void *ptr, int nr, int display_limit)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  init_candidate_win(uic);
  uic->cwin.is_active = 1;
  uic->cwin.candidate_index = -1;
  uic->cwin.nr_candidates = nr;
  uic->cwin.display_limit = display_limit;
  uic->cwin.layout_begin = 0;
  layout_candidate(uic);
  gtk_widget_show(uic->cwin.top_win);

  if (uic->win) {
    GdkWindow *toplevel;
    toplevel = gdk_window_get_toplevel(uic->win);
    gdk_window_add_filter(toplevel,
			  toplevel_window_candidate_cb,
			  uic);
  }
}


/* This function called by libuim.
   Selected by keyboard, etc. */
static void
cand_select_cb(void *ptr, int index)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  uic->cwin.candidate_index = index;
  layout_candidate(uic);
}

static void
cand_shift_page_cb(void *ptr, int direction)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;

  if(direction){
    if(uic->cwin.nr_candidates < uic->cwin.candidate_index + uic->cwin.display_limit )
      uic->cwin.candidate_index = 0;
    else
      uic->cwin.candidate_index += uic->cwin.display_limit;
  }
  else {
    if(uic->cwin.candidate_index - uic->cwin.display_limit < 0)
      uic->cwin.candidate_index = uic->cwin.nr_candidates - 1;
    else
      uic->cwin.candidate_index -= uic->cwin.display_limit;
  }
  layout_candidate(uic);
  uim_set_candidate_index(uic->uc, uic->cwin.candidate_index);
}

static void
cand_deactivate_cb(void *ptr)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  if(uic->cwin.top_win) {
    gtk_widget_hide(uic->cwin.top_win);
  }

  if (uic->win) {
    GdkWindow *toplevel;
    toplevel = gdk_window_get_toplevel(uic->win);
    gdk_window_remove_filter(toplevel, toplevel_window_candidate_cb, uic);
  }

  /*  uic->nr_psegs = 0; */
  uic->cwin.is_active = 0;
}


GtkIMContext *
im_module_create (const gchar *context_id)
{
  GObject *obj = g_object_new(type_im_uim, NULL);
  IMUIMContext *uic = IM_UIM_CONTEXT(obj);

  uic->win = NULL;
  uic->menu = NULL;
  uic->pseg = 0;
  uic->nr_psegs = 0;

  uic->cwin.top_win = NULL;
  uic->cwin.clist = NULL;
  uic->cwin.select_count = 0;

  if (!strncmp(context_id, "uim-", 4)) {
    context_id = &context_id[4];
  }

  uim_init();

  uic->uc =  uim_create_context(uic, "UTF-8",
				NULL, (char *)context_id,
				uim_iconv,
				im_uim_commit_string);
  if (uic->uc == NULL) {
    parent_class->finalize(obj);
    return NULL;
  }

  check_helper_connection();

  /**/

  uim_set_preedit_cb(uic->uc, clear_cb,
		     pushback_cb, update_cb);
  uim_set_prop_list_update_cb(uic->uc,
			      update_prop_list_cb);
  uim_set_prop_label_update_cb(uic->uc,
			      update_prop_label_cb);
  uim_set_candidate_selector_cb(uic->uc,
				cand_activate_cb,
				cand_select_cb,
				cand_shift_page_cb,
				cand_deactivate_cb);
  
  uim_prop_list_update(uic->uc);


  /* slave exists for using gtk+'s table based input method */
  uic->slave = g_object_new(GTK_TYPE_IM_CONTEXT_SIMPLE,
			    NULL);
  g_signal_connect(G_OBJECT(uic->slave), "commit",
		   G_CALLBACK(im_uim_commit_cb), uic);

  /**/
  uic->next = context_list.next;
  uic->prev = (IMUIMContext *)&context_list;
  context_list.next->prev = uic;
  context_list.next = uic;
  return GTK_IM_CONTEXT(uic);
}

void
im_module_list(GtkIMContextInfo ***contexts,
	       int *n_contexts)
{
  uim_context uc;
  int nr, i;
  static GtkIMContextInfo **info_list;

  uim_init();
  uc = uim_create_context(NULL, "UTF-8",
			  NULL, NULL, uim_iconv, NULL);
  if (uc == NULL)
    nr = 0;
  else
    nr = uim_get_nr_im(uc);

  info_list = malloc(sizeof(GtkIMContextInfo*)*nr);
  for (i = 0; i < nr; i++) {
    const char *name = uim_get_im_name(uc, i);
    const char *lang = uim_get_im_language(uc, i);
    char s[100];

    info_list[i] = malloc(sizeof(GtkIMContextInfo));

    sprintf(s, "uim-%s", name);
    info_list[i]->context_id = g_strdup(s);

    sprintf(s, "uim-%s (%s)", name, lang);
    info_list[i]->context_name = g_strdup(s);

    info_list[i]->domain = "gtk+";
    info_list[i]->domain_dirname = "";
    info_list[i]->default_locales = g_strdup(lang);
  }
  uim_release_context(uc);
  *contexts = info_list;
  *n_contexts = nr;
}

static void
im_uim_parse_helper_str(char *str)
{
  gchar **lines;
  
  if(focused_context) {
    if(strncmp("prop_list_get", str, 13) == 0){
      uim_prop_list_update(focused_context->uc);
    } else if(strncmp("prop_label_get", str, 14) == 0) {
      uim_prop_label_update(focused_context->uc);
    } else if(strncmp("prop_activate", str, 13) == 0) {
      lines = g_strsplit(str, "\n", 0);
      if(lines && lines[0]) {
	uim_prop_activate(focused_context->uc, lines[1]);
	g_strfreev(lines);
      }
    } else if(strncmp("focus_in", str, 8) == 0) {
      /*      focused_context = NULL; */
    }
  }
}


static void
helper_read_cb(gpointer p, int fd, GdkInputCondition c)
{
  char *tmp;

  uim_helper_read_proc(fd);
  while ((tmp = uim_helper_get_message())) {
    im_uim_parse_helper_str(tmp);
    free(tmp);
  }
}

static void
im_uim_helper_disconnect_cb(void)
{
  im_uim_fd = -1;
  gdk_input_remove(read_tag);
}

/* XXX:This is not a recommended way!! */
gboolean
uim_key_snoop(GtkWidget *grab_widget, GdkEventKey *key, gpointer data)
{
  if (focused_context) {
    int rv;
    int kv = convert_keyval(key->keyval);
    int mod = convert_modifier(key->state);
    
    if (key->type == GDK_KEY_RELEASE) {
      rv = uim_release_key(focused_context->uc, kv, mod);
    } else {
      rv = uim_press_key(focused_context->uc, kv, mod);
    }
    if (rv) {
      return FALSE;
    }
    return TRUE;
  }
  return FALSE;
}

void
im_module_init(GTypeModule *type_module)
{
  context_list.next = (IMUIMContext *)&context_list;
  context_list.prev = (IMUIMContext *)&context_list;
  type_im_uim =
    g_type_module_register_type(type_module,
				GTK_TYPE_IM_CONTEXT,
				"GtkIMContextUIM",
				&class_info, 0);

  /* XXX:This is not recommended way!! */
  snooper_id = gtk_key_snooper_install( (GtkKeySnoopFunc)uim_key_snoop, NULL );
}

void
im_module_exit()
{
  if (im_uim_fd != -1) {
    uim_helper_close_client_fd(im_uim_fd);
  }
  gtk_key_snooper_remove(snooper_id);
  uim_quit();
}
