;; SKK is a Japanese input method
;;
;; EUC-JP
;;
;; SKKϤϲξ֤ǹ
;; Following is list of SKK input state
;;  ľ direct
;;   kanji
;;  Ѵ converting
;;  ꤬ okuri
;;  ѿ latin
;;  ѱѿ wide-latin
;;
;; parent/child context is not used now
;;
(require "japanese.scm")
(require "generic-key.scm")

(define skk-dic-file-name "/usr/share/skk/SKK-JISYO.L")
(define skk-personal-dic-filename
  (string-append (getenv "HOME") "/.skk-jisyo"))
(define skk-uim-personal-dic-filename
  (string-append (getenv "HOME") "/.skk-uim-jisyo"))
(define skk-dic-init nil)
;; configs
(define skk-use-candidate-window? #t)
(define skk-candidate-op-count 0)
(define skk-use-recursive-learning? #t)
(define skk-commit-newline-explicitly? #f)  ;; turn into #t provided safe behavior
(define skk-style 'skk-style-ddskk-like)
(define skk-nr-candidate-max 10)

;; key defs
(define skk-latin-key
  (lambda (key key-state)
    (or
     (generic-off-key key key-state)
     (and (= (to-lower-char key) 108)
	  (not (modifier-key-mask key-state))))))
(define skk-wide-latin-key
  (lambda (key key-state)
    (and (= (to-lower-char key) 108)
	 (shift-key-mask key-state))))
(define skk-begin-conv-key
  (lambda (key key-state)
    (= key 32)))
(define skk-on-key
  (lambda (key key-state)
    (or
     (and (= (to-lower-char key) 106)
	  (control-key-mask key-state))
     (generic-on-key key key-state))))
(define skk-hankaku-kana-key
  (lambda (key key-state)
    (and
     (= (to-lower-char key) 113)
     (control-key-mask key-state))))
(define skk-return-key
  (lambda (key key-state)
    (generic-return-key key key-state)))
(define skk-commit-key
  (lambda (key key-state)
    (and (= (to-lower-char key) 106)
	 (control-key-mask key-state))))
(define skk-next-candidate-key
  (lambda (key key-state)
    (generic-next-candidate-key key key-state)))
(define skk-prev-candidate-key
  (lambda (key key-state)
    (or
     (generic-prev-candidate-key key key-state)
     (= (to-lower-char key) 120))))
(define skk-kana-toggle-key
  (lambda (key key-state)
    (and
     (= (to-lower-char key) 113)
     (not (modifier-key-mask key-state)))))
(define skk-cancel-key
  (lambda (key key-state)
    (generic-cancel-key key key-state)))

(define skk-backspace-key
  (lambda (key key-state)
    (generic-backspace-key key key-state)))
; 47 = '/'
(define skk-latin-conv-key
  (lambda (key key-state)
    (= key 47)))
; Q
(define skk-kanji-mode-key
  (lambda (key key-state)
    (= key 81)))

;; style elements
(define skk-preedit-attr-mode-mark)
(define skk-preedit-attr-head)
(define skk-preedit-attr-okuri)
(define skk-preedit-attr-pending-rk)
(define skk-preedit-attr-conv-body)
(define skk-preedit-attr-conv-okuri)
(define skk-preedit-attr-direct-pending-rk)
(define skk-preedit-attr-child-beginning-mark)
(define skk-preedit-attr-child-end-mark)
(define skk-preedit-attr-child-committed)
(define skk-child-context-beginning-mark)
(define skk-child-context-end-mark)
(define skk-show-cursor-on-preedit?)
(define skk-show-candidates-with-okuri?)
;; style specification
(define skk-style-spec
  '(;; (style-element-name . validator)
    (skk-preedit-attr-mode-mark            . preedit-attr?)
    (skk-preedit-attr-head                 . preedit-attr?)
    (skk-preedit-attr-okuri                . preedit-attr?)
    (skk-preedit-attr-pending-rk           . preedit-attr?)
    (skk-preedit-attr-conv-body            . preedit-attr?)
    (skk-preedit-attr-conv-okuri           . preedit-attr?)
    (skk-preedit-attr-direct-pending-rk    . preedit-attr?)
    (skk-preedit-attr-child-beginning-mark . preedit-attr?)
    (skk-preedit-attr-child-end-mark       . preedit-attr?)
    (skk-preedit-attr-child-committed      . preedit-attr?)
    (skk-child-context-beginning-mark      . string?)
    (skk-child-context-end-mark            . string?)
    (skk-show-cursor-on-preedit?           . boolean?)
    (skk-show-candidates-with-okuri?       . boolean?)))
;; predefined styles
(define skk-style-uim
  '((skk-preedit-attr-mode-mark            . preedit-reverse)
    (skk-preedit-attr-head                 . preedit-reverse)
    (skk-preedit-attr-okuri                . preedit-reverse)
    (skk-preedit-attr-pending-rk           . preedit-reverse)
    (skk-preedit-attr-conv-body            . preedit-reverse)
    (skk-preedit-attr-conv-okuri           . preedit-reverse)
    (skk-preedit-attr-direct-pending-rk    . preedit-underline)
    (skk-preedit-attr-child-beginning-mark . preedit-reverse)
    (skk-preedit-attr-child-end-mark       . preedit-reverse)
    (skk-preedit-attr-child-committed      . preedit-reverse)
    (skk-child-context-beginning-mark      . "[")
    (skk-child-context-end-mark            . "]")
    (skk-show-cursor-on-preedit?           . #f)
    (skk-show-candidates-with-okuri?       . #t)))
(define skk-style-ddskk-like
  '((skk-preedit-attr-mode-mark            . preedit-underline)
    (skk-preedit-attr-head                 . preedit-underline)
    (skk-preedit-attr-okuri                . preedit-underline)
    (skk-preedit-attr-pending-rk           . preedit-underline)
    (skk-preedit-attr-conv-body            . preedit-reverse)
    (skk-preedit-attr-conv-okuri           . preedit-underline)
    (skk-preedit-attr-direct-pending-rk    . preedit-underline)
    (skk-preedit-attr-child-beginning-mark . preedit-underline)
    (skk-preedit-attr-child-end-mark       . preedit-underline)
    (skk-preedit-attr-child-committed      . preedit-underline)
    (skk-child-context-beginning-mark      . "")
    (skk-child-context-end-mark            . "")
    (skk-show-cursor-on-preedit?           . #t)
    (skk-show-candidates-with-okuri?       . #f)))

;; access

(define skk-context-commit-raw
  (lambda (c)
    (car (nthcdr 14 c))))
(define skk-context-set-commit-raw!
  (lambda (c raw)
    (set-car! (nthcdr 14 c) raw)))
(define skk-context-latin-conv
  (lambda (c)
    (car (nthcdr 13 c))))
(define skk-context-set-latin-conv!
  (lambda (c lc)
    (set-car! (nthcdr 13 c) lc)))
(define skk-context-editor
  (lambda (c)
    (car (nthcdr 12 c))))
(define skk-context-set-editor!
  (lambda (c cnt)
    (set-car! (nthcdr 12 c) cnt)))
(define skk-context-parent-context
  (lambda (c)
    (car (nthcdr 11 c))))
(define skk-context-set-parent-context!
  (lambda (c cnt)
    (set-car! (nthcdr 11 c) cnt)))
(define skk-context-child-context
  (lambda (c)
    (car (nthcdr 10 c))))
(define skk-context-set-child-context!
  (lambda (c cnt)
    (set-car! (nthcdr 10 c) cnt)))
(define skk-context-candidate-window
  (lambda (c)
    (car (nthcdr 9 c))))
(define skk-context-set-candidate-window!
  (lambda (c cnt)
    (set-car! (nthcdr 9 c) cnt)))
(define skk-context-candidate-op-count
  (lambda (c)
    (car (nthcdr 8 c))))
(define skk-context-set-candidate-op-count!
  (lambda (c cnt)
    (set-car! (nthcdr 8 c) cnt)))
(define skk-context-rk-context
  (lambda (c)
    (car (nthcdr 7 c))))
(define skk-context-set-rk-context!
  (lambda (c rkc)
    (set-car! (nthcdr 7 c) rkc)))
(define skk-context-nth
  (lambda (c)
    (car (nthcdr 6 c))))
(define skk-context-set-nth!
  (lambda (c nth)
    (set-car! (nthcdr 6 c) nth)))
(define skk-context-okuri
  (lambda (c)
    (car (nthcdr 4 c))))
(define skk-context-set-okuri!
  (lambda (c tail)
    (set-car! (nthcdr 4 c) tail)))
(define skk-context-okuri-head
  (lambda (c)
    (car (nthcdr 3 c))))
(define skk-context-set-okuri-head!
  (lambda (c okuri-head)
    (set-car! (nthcdr 3 c) okuri-head)))
(define skk-context-set-head!
  (lambda (c head)
    (set-car! (nthcdr 2 c) head)))
(define skk-context-head
  (lambda (c)
    (car (nthcdr 2 c))))
(define skk-context-kana-mode
  (lambda (sc)
    (car (nthcdr 1 sc))))
(define skk-context-set-kana-mode!
  (lambda (sc mode)
    (set-car! (nthcdr 1 sc) mode)))
(define skk-context-state
  (lambda (c)
    (car (nthcdr 0 c))))
(define skk-context-set-state!
  (lambda (c s)
    (set-car! (nthcdr 0 c) s)))
;; state kana head okuri tail candidates nth rk

(define skk-type-hiragana 0)
(define skk-type-katakana 1)
(define skk-type-hankana 2)

(define skk-mode-latin 0)
(define skk-mode-hiragana 1)
(define skk-mode-katakana 2)
(define skk-mode-wide-latin 3)
(define skk-mode-hankana 4)

(define skk-find-root-context
  (lambda (sc)
    (let ((pc (skk-context-parent-context sc)))
      (if pc
	  (skk-find-root-context pc)
	  sc))))

(define skk-find-descendant-context
  (lambda (sc)
    (let ((csc (skk-context-child-context sc)))
      (if csc
	  (skk-find-descendant-context csc)
	  sc))))

(define skk-read-personal-dictionary
  (lambda ()
    (or (skk-lib-read-personal-dictionary skk-uim-personal-dic-filename)
	(skk-lib-read-personal-dictionary skk-personal-dic-filename))))

(define skk-save-personal-dictionary
  (lambda ()
    (skk-lib-save-personal-dictionary skk-uim-personal-dic-filename)))

(define skk-current-context
  (lambda (c)
    (skk-find-descendant-context (context-data c))))

(define skk-flush
  (lambda (sc)
    (rk-flush (skk-context-rk-context sc))
    (if skk-use-recursive-learning?
	(skk-editor-flush (skk-context-editor sc)))
    (skk-context-set-state! sc 'skk-state-direct)
    (skk-context-set-head! sc '())
    (skk-context-set-okuri-head! sc "")
    (skk-context-set-okuri! sc '())
    (skk-context-set-candidate-op-count! sc 0)
    (skk-context-set-candidate-window! sc #f)
    (skk-context-set-latin-conv! sc #f)))

(define skk-context-new
  (lambda (id)
    (if (null? skk-dic-init)
	(begin
	  (set! skk-dic-init #t)
	  (if skk-use-recursive-learning?
	   (require "skk-editor.scm"))
	  (skk-lib-dic-open skk-dic-file-name)
	  (skk-read-personal-dictionary)))
    (let ((sc 
	   (copy-list 
	    '(skk-state-latin 0 "" "" "" () () () () () () () () () #f))))
      (skk-context-set-head! sc ())
      (skk-context-set-rk-context! sc
				   (rk-context-new ja-rk-rule #t #f))
      (skk-context-set-child-context! sc ())
      (skk-context-set-parent-context! sc ())
      (if skk-use-recursive-learning?
	  (skk-context-set-editor! sc
				   (skk-editor-new sc id)))
      (skk-flush sc)
      (skk-context-set-state! sc 'skk-state-latin)
      sc)))

(define skk-make-string
  (lambda (sl kana)
    (let ((get-str-by-type 
	   (lambda (l)
	     (cond
	      ((= kana skk-type-hiragana)
	       (caar l))
	      ((= kana skk-type-katakana)
	       (car (cdar l)))
	      ((= kana skk-type-hankana)
	       (cadr (cdar l)))))))
    (if sl
	(string-append (skk-make-string (cdr sl) kana)
		       (get-str-by-type sl))
	""))))

(define skk-opposite-kana
  (lambda (kana)
    (cond
     ((= kana skk-type-hiragana)
      skk-type-katakana)
     ((= kana skk-type-katakana)
      skk-type-hiragana)
     ((= kana skk-type-hankana)
      skk-type-hiragana))))  ; different to ddskk's behavior

(define skk-context-kana-toggle
  (lambda (sc)
    (let* ((kana (skk-context-kana-mode sc))
	   (opposite-kana (skk-opposite-kana kana)))
      (skk-context-set-kana-mode! sc opposite-kana))))

(define skk-get-string-mode-part
  (lambda (sc res)
    (let* ((type (skk-context-kana-mode sc))
	   (get-str-by-type 
	    (lambda (l)
	      (cond
	       ((= type skk-type-hiragana)
		(car l))
	       ((= type skk-type-katakana)
		(car (cdr l)))
	       ((= type skk-type-hankana)
		(cadr (cdr l)))))))
      (get-str-by-type res))))

(define skk-do-get-string-by-mode
  (lambda (sc str)
    (if str
	(if (string? (car str))
	    (skk-get-string-mode-part sc str)
	    (string-append
	     (skk-do-get-string-by-mode sc (car str))
	     (skk-do-get-string-by-mode sc (cdr str))))
	"")))

(define skk-get-string-by-mode
  (lambda (sc str)
    (let ((res (skk-do-get-string-by-mode sc str)))
      (if (and res (> (length res) 0))
	  res
	  nil))))

(define skk-get-nth-candidate
  (lambda (sc n)
    (skk-lib-get-nth-candidate
     n
     (skk-make-string (skk-context-head sc) skk-type-hiragana)
     (skk-context-okuri-head sc)
     (skk-make-string (skk-context-okuri sc) skk-type-hiragana))))

(define skk-get-current-candidate
  (lambda (sc)
    (skk-get-nth-candidate
     sc
     (skk-context-nth sc))))

(define skk-commit-raw
  (lambda (sc id key key-state)
    (let ((psc (skk-context-parent-context sc)))
      (if psc
	  (skk-editor-commit-raw
	   (skk-context-editor psc)
	   key key-state)
	  (begin
	    (skk-context-set-commit-raw! sc #f)
	    (im-commit-raw id))))))

;; commit string
(define skk-commit
  (lambda (sc id str)
    (let ((psc (skk-context-parent-context sc)))
      (if psc
	  (skk-editor-commit
	   (skk-context-editor psc) (skk-lib-remove-annotation str))
	  (im-commit id (skk-lib-remove-annotation str))))))

(define skk-prepare-commit-string
  (lambda (sc id)
    (let* ((cand (skk-get-current-candidate sc))
	   (okuri (skk-make-string (skk-context-okuri sc)
				   (skk-context-kana-mode sc)))
	   (res (string-append cand okuri)))
      (skk-lib-commit-candidate
       (skk-make-string (skk-context-head sc) skk-type-hiragana)
       (skk-context-okuri-head sc)
       (skk-make-string (skk-context-okuri sc) skk-type-hiragana)
       (skk-context-nth sc))
      (if (> (skk-context-nth sc) 0)
	  (skk-save-personal-dictionary))
      (skk-reset-candidate-window sc id)
      (skk-flush sc)
      (skk-context-set-state! sc 'skk-state-direct)
      (skk-update-mode id sc)
      res)))


(define skk-append-string
  (lambda (sc str)
    (and
     str
     (if (not (string? (car str)))
	 (begin
	   (skk-append-string sc (car str))
	   (skk-append-string sc (cdr str))
	   )
	 #t)
     (skk-context-set-head!
      sc
      (cons str (skk-context-head sc))))))

(define skk-begin-conversion
  (lambda (sc id)
    (let ((res))
      ;; get residual 'n'
      (if (= (skk-context-state sc) 'skk-state-kanji)
 	  (let ((tmp (rk-push-key-last! (skk-context-rk-context sc))))
 	    (if tmp
 		(skk-append-string sc tmp))))
      ;;
      (set! res
	    (skk-lib-get-entry
	     (skk-make-string (skk-context-head sc) skk-type-hiragana)
	     (skk-context-okuri-head sc)
	     (skk-make-string (skk-context-okuri sc) skk-type-hiragana)))
      (if res
	  (begin
	    (skk-context-set-nth! sc 0)
	    (skk-context-set-state!
	     sc 'skk-state-converting))
	  (if skk-use-recursive-learning?
	      (skk-setup-child-context sc id)
	      (skk-flush sc)))
      ())))

(define skk-do-update-preedit
  (lambda (id sc)
    (let ((rkc (skk-context-rk-context sc))
	  (stat (skk-context-state sc))
	  (csc (skk-context-child-context sc)))
      (if (and
	   (not csc)
	   (or
	    (= stat 'skk-state-kanji)
	    (= stat 'skk-state-okuri)))
	  (im-pushback-preedit id skk-preedit-attr-mode-mark ""))
      (if (or
	   csc
	   (= stat 'skk-state-converting))
	  (im-pushback-preedit id skk-preedit-attr-mode-mark ""))
      (if (or
	   (= stat 'skk-state-kanji)
	   (= stat 'skk-state-okuri)
	   (and
	    csc
	    (= stat 'skk-state-converting)))
	  (let ((h (skk-make-string 
		    (skk-context-head sc)
		    (skk-context-kana-mode sc))))
	    (if (string? h)
		(im-pushback-preedit
		 id skk-preedit-attr-head
		 h))))
      (if (and
	   (= stat 'skk-state-converting)
	   (not csc))
	  (begin
	    (im-pushback-preedit
	     id
	     (bit-or skk-preedit-attr-conv-body
		     preedit-cursor)
	     (skk-get-current-candidate sc))
	    (im-pushback-preedit
	     id skk-preedit-attr-conv-okuri
	     (skk-make-string (skk-context-okuri sc)
			      (skk-context-kana-mode sc)))))

      (if (or
	   (= stat 'skk-state-okuri)
	   (and
	    csc
	    (= stat 'skk-state-converting)
	    (skk-context-okuri sc)))
	  (begin
	    (im-pushback-preedit 
	     id skk-preedit-attr-okuri
	     (string-append
	      "*" (skk-make-string (skk-context-okuri sc)
				   (skk-context-kana-mode sc))))))

      (if (or
	   (= stat 'skk-state-direct)
	   (= stat 'skk-state-latin)
	   (= stat 'skk-state-wide-latin))
	  (begin
	    (im-pushback-preedit id skk-preedit-attr-direct-pending-rk
				 (rk-pending rkc))
	    (im-pushback-preedit id preedit-cursor ""))
	  (begin
	    (im-pushback-preedit id skk-preedit-attr-pending-rk
				 (rk-pending rkc))
	    (if (and
		 (or
		  (= stat 'skk-state-kanji)
		  (= stat 'skk-state-okuri))
		 skk-show-cursor-on-preedit?)
		(im-pushback-preedit id preedit-cursor ""))))

      ;; child context's preedit
      (if csc
	  (let ((editor (skk-context-editor sc)))
	    (im-pushback-preedit id skk-preedit-attr-child-beginning-mark
				 skk-child-context-beginning-mark)
	    (im-pushback-preedit id skk-preedit-attr-child-committed
				 (skk-editor-get-left-string editor))
	    (skk-do-update-preedit id csc)
	    (im-pushback-preedit id skk-preedit-attr-child-committed
				 (skk-editor-get-right-string editor))
	    (im-pushback-preedit id skk-preedit-attr-child-end-mark
				 skk-child-context-end-mark)
	    )))))

(define skk-update-mode
  (lambda (id sc)
    (let ((stat (skk-context-state sc))
	  (mode)
	  (kana (skk-context-kana-mode sc)))
      (set! mode
	    (cond
	     ((= kana skk-type-hiragana)
	      skk-mode-hiragana)
	     ((= kana skk-type-katakana)
	      skk-mode-katakana)
	     ((= kana skk-type-hankana)
	      skk-mode-hankana)))
      (if (= stat 'skk-state-latin)
	  (set! mode skk-mode-latin))
      (if (= stat 'skk-state-wide-latin)
	  (set! mode skk-mode-wide-latin))
      (im-update-mode id mode))))

(define skk-update-preedit
  (lambda (id sc)
    (if (not (skk-context-commit-raw sc))
	(begin
	  (im-clear-preedit id)
	  (skk-do-update-preedit id (skk-find-root-context sc))
	  (im-update-preedit id))
	(skk-context-set-commit-raw! sc #f))))


;; called from skk-editor
(define skk-commit-editor-context
  (lambda (sc id str)
    (skk-flush sc)
    (skk-context-set-child-context! sc nil)
    (skk-commit sc id str)))

(define skk-proc-state-direct-no-preedit
  (lambda (key key-state id sc rkc)
    (and
     (if (skk-cancel-key key key-state)
	 (begin
	   (skk-commit-raw sc id key key-state)
	   #f)
	 #t)
     (if (skk-wide-latin-key key key-state)
	 (begin
	   (skk-context-set-state! sc 'skk-state-wide-latin)
	   (rk-flush rkc)
	   (skk-update-mode id sc)
	   (skk-update-prop-label sc id)
	   #f)
	 #t)
     (if (skk-latin-key key key-state)
	 (begin
	   (skk-context-set-state! sc 'skk-state-latin)
	   (rk-flush rkc)
	   (skk-update-mode id sc)
	   (skk-update-prop-label sc id)
	   #f)
	 #t)
     (if (skk-latin-conv-key key key-state)
	 (begin
	   (skk-context-set-state! sc 'skk-state-kanji)
	   (skk-update-mode id sc)
	   (skk-context-set-latin-conv! sc #t)
	   #f)
	 #t)
     (if (skk-kanji-mode-key key key-state)
	 (begin
	   (skk-context-set-state! sc 'skk-state-kanji)
	   (skk-update-mode id sc)
	   (skk-context-set-latin-conv! sc #f)
	   #f)
	 #t)
       (if (skk-hankaku-kana-key key key-state)
	   (begin
	     (skk-context-set-kana-mode! sc skk-type-hankana)
	     (skk-update-mode id sc)
	     (skk-update-prop-label sc id)
	     #f)
	   #t)
     (if (skk-kana-toggle-key key key-state)
	 (begin 
	   (skk-context-kana-toggle sc)
	   (skk-update-mode id sc)
	   (skk-update-prop-label sc id)
	   #f)
	 #t)
     #t)))

(define skk-proc-state-direct
  (lambda (c key key-state)
    (let* ((sc (skk-current-context c))
	   (id (context-id c))
	   (key-str (charcode->string (to-lower-char key)))
	   (rkc (skk-context-rk-context sc))
	   (res))
      (set! res nil)
      (and
       ;; at first, no preedit mode
       (if (string-equal? (rk-pending rkc) "")
	   (skk-proc-state-direct-no-preedit key key-state id sc rkc)
	   #t)
       (if (skk-cancel-key key key-state)
	   (begin
	     (skk-flush sc)
	     #f)
	   #t)
       (if (skk-backspace-key key key-state)
	   (if (not (rk-backspace rkc))
	       (begin
		 (skk-commit-raw sc id key key-state)
		 #f)
	       #f)
	   #t)
       (if (or
	    (control-key-mask key-state)
	    (= key 32))
	   (begin
	     (skk-commit-raw sc id key key-state)
	     #f)
	   #t)
       (if (and
	    (shift-key-mask key-state)
	    (alphabet-char? key))
	   (begin
	     (skk-context-set-state! sc 'skk-state-kanji)
	     (skk-update-mode id sc)
	     (set! key (to-lower-char key))
	     (set! key-str (charcode->string key))
	     #t)
	   #t)
       (if (and
	    (not (skk-context-head sc))
	    (not (string-find (rk-expect rkc) key-str))
	    (not (rk-pending rkc)))
	   (begin
	     (skk-commit-raw sc id key key-state)
	     (skk-flush sc)
	     (skk-update-preedit id sc)
	     #f)
	   #t)
       (if (symbol? key)
	   (begin
	     (skk-flush sc)
	     (skk-commit-raw sc id key key-state)
	     #f)
	   #t)
       (begin
	 (set! res
	       (rk-push-key!
		rkc
		key-str))
	 #t));;and
      ;; update state
      (if (= (skk-context-state sc) 'skk-state-kanji)
	  (if res
	      (skk-append-string sc res)
	      (set! res ())))
      (if (= (skk-context-state sc) 'skk-state-direct)
	  (skk-get-string-by-mode sc res)
	  nil))))

(define skk-proc-state-kanji
  (lambda (c key key-state)
    (let* ((sc (skk-current-context c))
	   (id (context-id c))
	   (rkc (skk-context-rk-context sc))
	   (stat (skk-context-state sc))
	   (res))
      (and
       (if (skk-begin-conv-key key key-state)
	   (begin
	     (skk-begin-conversion sc id)
	     #f)
	   #t)
       (if (skk-cancel-key key key-state)
	   (begin
	     (skk-flush sc)
	     #f)
	   #t)
       (if (skk-backspace-key key key-state)
	   (begin
	     (if (not (rk-backspace rkc))
		 (if (> (length (skk-context-head sc)) 0)
		     (skk-context-set-head!
		      sc (cdr (skk-context-head sc)))
		     (begin
		       (skk-context-set-state! sc 'skk-state-direct)
		       (skk-flush sc))))
	     #f)
	   #t)
       (if (or
	    (skk-commit-key key key-state)
	    (skk-return-key key key-state))
	   (begin
	     (skk-commit sc id (skk-make-string
				(skk-context-head sc)
				(skk-context-kana-mode sc)))
	     (skk-flush sc)
	     (skk-context-set-state! sc 'skk-state-direct)
	     (skk-update-mode id sc)
 	     (if (skk-return-key key key-state)
		 (if skk-commit-newline-explicitly?
		     (skk-commit sc id "\n")
		     (skk-proc-state-direct c key key-state)))
 	     #f)
	   #t)
       (if (skk-context-latin-conv sc)
          (begin
            (if (usual-char? key)
                (let* ((s (charcode->string key))
                       (p (cons s s)))
                  (skk-append-string sc p)))
            #f)
          #t)
       (if (and (shift-key-mask key-state)
		(skk-context-head sc))
	   (begin
	     (skk-context-set-state! sc 'skk-state-okuri)
	     (skk-update-mode id sc)
	     (set! key (to-lower-char key))
	     (skk-context-set-okuri-head! sc
					  (charcode->string key))
	     (let ((tmp (rk-push-key-last! (skk-context-rk-context sc))))
	       (if tmp
		   (skk-append-string sc tmp)))
	     #t)
	   #t)
       (if (skk-kana-toggle-key key key-state)
	   (begin
	     (if (skk-context-head sc)
		 (skk-commit sc id (skk-make-string
				    (skk-context-head sc)
				    (skk-opposite-kana
				     (skk-context-kana-mode sc)))))
            (skk-flush sc)
            (skk-context-set-state! sc 'skk-state-direct)
            (skk-update-mode id sc)
            #f)
          #t)
       (begin
	 (set! key (to-lower-char key))  
	 (set! stat (skk-context-state sc))
	 (set! res
	       (rk-push-key!
		rkc
		(charcode->string key)))
	 (if (and res (= stat 'skk-state-kanji))
	     (begin
	       (skk-context-set-head! sc
				      (cons
				       res
				       (skk-context-head sc)))))
	 (if (and res (= stat 'skk-state-okuri))
	     (begin
	       (skk-context-set-okuri! sc
				       (cons res ()))
	       (skk-begin-conversion sc id)))))
      nil)))

(define skk-setup-child-context
  (lambda (sc id)
    (let ((csc (skk-context-new id)))
      (skk-context-set-child-context! sc csc)
      (skk-context-set-parent-context! csc sc)
      (skk-context-set-state! csc 'skk-state-direct))))

(define skk-check-candidate-window-begin
  (lambda (sc id)
    (if
     (and
      (not
       (skk-context-candidate-window sc))
      skk-use-candidate-window?
      (> (skk-context-candidate-op-count sc)
	 skk-candidate-op-count))
     (begin
       (skk-context-set-candidate-window! sc #t)
       (im-begin-candidate
	id
	(skk-lib-get-nr-candidates
	 (skk-make-string (skk-context-head sc) skk-type-hiragana)
	 (skk-context-okuri-head sc)
	 (skk-make-string (skk-context-okuri sc) skk-type-hiragana))
	skk-nr-candidate-max)))))

(define skk-change-candidate-index
  (lambda (sc id incr)
    (if incr
	(begin
	  (skk-context-set-nth! sc
				(+ 1 (skk-context-nth sc)))
	  (skk-context-set-candidate-op-count!
	   sc
	   (+ 1 (skk-context-candidate-op-count sc))))
	(begin
	  (if (> (skk-context-nth sc) 0)
	      (skk-context-set-nth! sc (- (skk-context-nth sc) 1))
	      (skk-context-set-nth! sc (- (skk-lib-get-nr-candidates
					   (skk-make-string
					    (skk-context-head sc)
					    skk-type-hiragana)
					   (skk-context-okuri-head sc)
					   (skk-make-string
					    (skk-context-okuri sc)
					    skk-type-hiragana))
					  1)))))
    (if (not (skk-get-current-candidate sc))
	(begin
	  (skk-context-set-nth! sc 0)
	  (if skk-use-recursive-learning?
	      (begin
		(skk-reset-candidate-window sc id)
		(skk-setup-child-context sc id)))))
    (if (not (skk-context-child-context sc))
	(begin
	  ;; Windowɽ򳫻Ϥ뤫
	  (skk-check-candidate-window-begin sc id)
	  ;;
	  (if (skk-context-candidate-window sc)
	      (im-update-candidate id (skk-context-nth sc)))))
    #f))

(define skk-reset-candidate-window
  (lambda (sc id)
    (if (skk-context-candidate-window sc)
	(begin
	  (im-end-candidate id sc)
	  (skk-context-set-candidate-window! sc #f)))
    (skk-context-set-candidate-op-count! sc 0)))

(define skk-back-to-kanji-state
  (lambda (sc id)
    (skk-reset-candidate-window sc id)
    (skk-context-set-state! sc 'skk-state-kanji)
    (skk-context-set-okuri-head! sc "")
    (if (car (skk-context-okuri sc))
	(skk-context-set-head! sc
			       (cons (car (skk-context-okuri sc))
				     (skk-context-head sc))))
    (skk-context-set-okuri! sc ())))

(define skk-proc-state-converting
  (lambda (c key key-state)
    (let ((sc (skk-current-context c))
	  (id (context-id c))
	  (res ()))
      (and
       (if (skk-next-candidate-key key key-state)
	   (skk-change-candidate-index sc id #t)
	   #t)
       (if (skk-prev-candidate-key key key-state)
	   (skk-change-candidate-index sc id #f)
	   #t)
       (if (skk-cancel-key key key-state)
	   (begin
	     ;; back to kanji state
	     (skk-back-to-kanji-state sc id)
	     #f)
	   #t)
       (if (or
	    (skk-commit-key key key-state)
	    (skk-return-key key key-state))
	   (begin
	     (set! res (skk-prepare-commit-string sc id))
	     (if (skk-return-key key key-state)
		 (begin
		   (skk-commit sc id res)
		   (set! res ())
		   (if skk-commit-newline-explicitly?
		       (skk-commit sc id "\n")
		       (skk-proc-state-direct c key key-state))))
	     #f)
	   #t)
       (begin
	 (skk-context-set-state! sc 'skk-state-direct)
	 (skk-update-mode id sc)
	 (set! res (skk-get-current-candidate sc))
	 (skk-reset-candidate-window sc id)
	 (set!
	  res
	  (string-append res 
			 (skk-make-string
			  (skk-context-okuri sc)
			  (skk-context-kana-mode sc))))
	 (skk-flush sc)
	 (let ((res2 (skk-proc-state-direct c key key-state)))
	   (set!
	    res
	    (string-append res 
			   (skk-make-string
			    (skk-context-okuri sc)
			    (skk-context-kana-mode sc))))
	   (if (string? res2)
	       (set! res
		     (string-append res res2))))))
      res)))

(define skk-proc-state-okuri
  (lambda (c key key-state)
    (let* ((sc (skk-current-context c))
	   (rkc (skk-context-rk-context sc))
	   (id (context-id c))	   
	   (res))
      (and
       (if (skk-cancel-key key key-state)
	   (begin
	     (rk-flush rkc)
	     (skk-context-set-state! sc 'skk-state-kanji)
	     #f)
	   #t)
       (if (skk-backspace-key key key-state)
	   (begin
	     (if (not (rk-backspace rkc))
		 (begin
		   (if (cdr (skk-context-okuri sc))
		       (skk-context-set-okuri! sc
			(cdr (skk-context-okuri sc)))
		       (begin
			 (skk-context-set-okuri! sc '())
		       (skk-context-set-state! sc 'skk-state-kanji)))))
	     #f)
	   #t)
       ;; committing incomplete head: conformed the behavior to ddskk
       (if (or
	    (skk-commit-key key key-state)
	    (skk-return-key key key-state))
	   (begin
	     (skk-commit sc id (skk-make-string
				(skk-context-head sc)
				(skk-context-kana-mode sc)))
	     (skk-flush sc)
	     (skk-context-set-state! sc 'skk-state-direct)
	     (skk-update-mode id sc)
	     (if (skk-return-key key key-state)
		 (skk-proc-state-direct c key key-state))
	     #f)
	   #t)
       (begin
	 (set! res
	       (rk-push-key!
		rkc
		(charcode->string (to-lower-char key))))
	 (if res
	     (begin
	       (skk-context-set-okuri!
		sc
		(cons res (skk-context-okuri sc)))
	       (if (string-equal? (rk-pending rkc) "")
		   (skk-begin-conversion sc id))))))
      ())))

(define skk-proc-state-latin
  (lambda (c key key-state)
    (let ((sc (skk-current-context c))
	  (id (context-id c)))
      (if
       (skk-on-key key key-state)
       (begin
	 (skk-context-set-state! sc 'skk-state-direct)
	 (skk-update-mode (context-id c) sc)
	 (skk-update-prop-label sc id))
       (skk-commit-raw sc (context-id c) key key-state))
      ())))

(define skk-proc-state-wide-latin
  (lambda (c key key-state)
    (let* ((w (ja-wide (charcode->string key)))
	   (id (context-id c))
	   (sc (skk-current-context c)))
      (and
       (if (skk-on-key key key-state)
	   (begin
	     (skk-flush sc)
	     (skk-update-mode id sc)
	     (skk-update-prop-label sc id)
	     #f)
	   #t)
       (if (and (modifier-key-mask key-state)
		(not (shift-key-mask key-state)))
	   (begin
	     (skk-commit-raw sc id key key-state)
	     #f)
	   #t)
       (if w
	   (skk-commit sc id w)
	   (skk-commit-raw sc id key key-state)))
      ())))

(define skk-push-key
  (lambda (c id key key-state)
    (let* ((sc (skk-current-context c))
	   (state (skk-context-state sc))
	   (fun)
	   (res))
      (if (= state 'skk-state-direct)
	  (set! fun skk-proc-state-direct))
      (if (= state 'skk-state-kanji)
	  (set! fun skk-proc-state-kanji))
      (if (= state 'skk-state-converting)
	  (set! fun skk-proc-state-converting))
      (if (= state 'skk-state-okuri)
	  (set! fun skk-proc-state-okuri))
      (if (= state 'skk-state-latin)
	  (set! fun skk-proc-state-latin))
      (if (= state 'skk-state-wide-latin)
	  (set! fun skk-proc-state-wide-latin))
      (set! res (fun c key key-state))
      (if res
	  (skk-commit sc id res))
      (skk-update-preedit id sc)
      )))

(define skk-init-handler
  (lambda (id arg)
    (let* ((c (find-context id)))
      (set-context-data! c
			 (skk-context-new id))
      (im-clear-mode-list id)
      (im-pushback-mode-list id "ľ")
      (im-pushback-mode-list id "Ҥ餬")
      (im-pushback-mode-list id "")
      (im-pushback-mode-list id "ѱѿ")
      (im-pushback-mode-list id "Ⱦѥ")
      (im-update-mode-list id)
      (im-update-mode id skk-mode-latin)
      (skk-update-prop-list id)
      (update-style skk-style-spec (symbol-value skk-style)))))

(define skk-press-key-handler
  (lambda (id key state)
    (let ((c (find-context id)))
      (if (and (number? key) (< key 32))
	  (im-commit-raw id)
	  (skk-push-key c id key state)))))

(define skk-release-key-handler
  (lambda (id key state)
    (let* ((c (find-context id))
	   (sc (skk-current-context c))
	   (state (skk-context-state sc)))
      (if (= state 'skk-state-latin)
	  ;; don't eat key release event for apps
	  (begin
	    (skk-context-set-commit-raw! sc #f)
	    (im-commit-raw id))))))

(define skk-reset-handler
  (lambda (id)
    ()))

(define skk-mode-handler
  (lambda (id mode)
    (let* ((c (find-context id))
	   (sc (context-data c)))
      (skk-flush sc)
      (if (= mode skk-mode-latin)
	  (skk-context-set-state! sc 'skk-state-latin))
      (if (= mode skk-mode-hiragana)
	  (begin
	    (skk-context-set-state! sc 'skk-state-direct)
	    (skk-context-set-kana-mode! sc skk-type-hiragana)))
      (if (= mode skk-mode-katakana)
	  (begin
	    (skk-context-set-state! sc 'skk-state-direct)
	    (skk-context-set-kana-mode! sc skk-type-katakana)))
      (if (= mode skk-mode-wide-latin)
	  (skk-context-set-state! sc 'skk-state-wide-latin))
      (if (= mode skk-mode-hankana)
	  (begin
	    (skk-context-set-state! sc 'skk-state-direct)
	    (skk-context-set-kana-mode! sc skk-type-hankana)))
      (skk-update-preedit id sc)
      ())))

(define skk-get-candidate-handler
  (lambda (id idx)
    (let* ((c (find-context id))
	   (sc (context-data c))
	   (dcsc (skk-find-descendant-context sc))
	   (cand (skk-get-nth-candidate dcsc idx))
	   (okuri (skk-context-okuri dcsc)))
      (list
       (if (and
	    okuri
	    skk-show-candidates-with-okuri?)
	   (string-append cand
			  (skk-make-string okuri skk-type-hiragana))
	   cand)
       (digit->string (+ idx 1))))))

(define skk-set-candidate-index-handler
  (lambda (id idx)
    (let* ((c (find-context id))
	   (sc (skk-current-context c)))
      (skk-context-set-nth! sc idx)
      (skk-commit sc id (skk-prepare-commit-string sc id))
      (skk-update-preedit  id sc))))

(define skk-prop-handler
  (lambda (id message)
    (let* ((c (find-context id))
	   (sc (context-data c)))
      (skk-flush sc)
      (skk-update-preedit sc id)
    (if (string-equal? message
		       "prop_skk_hiragana")
	(begin
	  (im-update-mode id 1)
	  (skk-context-set-state! sc #t)))
    (if (string-equal? message
		       "prop_skk_katakana")
	(begin
	  (im-update-mode id 2)
	  (skk-context-set-on! sc #t)))
    (if (string-equal? message
		       "prop_skk_latin")
	(begin
	  (im-update-mode id 0)
	  (skk-context-set-state! sc 'skk-state-latin)))
    (if (string-equal? message
		       "prop_skk_wide_latin")
	(begin
	  (im-update-mode id 3)
	  (skk-context-set-state! sc 'skk-state-wide-latin)))
    (skk-update-prop-label sc id))))


(define skk-update-prop-label
  (lambda (sc id)
    (let* ((state (skk-context-state sc))
	   (kana (skk-context-kana-mode sc))
	   (str ""))
      (cond
       ((= state 'skk-state-latin)
	(set! str "S\tľ\n"))
       ((= state 'skk-state-wide-latin)
	(set! str "\tѱѿ\n"))
       ((= kana skk-type-hiragana)
	(set! str "\tҤ餬\n"))
       ((= kana skk-type-katakana)
	(set! str "\t\n"))
       ((= kana skk-type-hankana)
	(set! str "\tȾѥ\n")))
      (im-update-prop-label id str))))

(define skk-update-prop-list
  (lambda (id) 
    (let* ((c (find-context id))
	   (sc (context-data c))
	   (state (skk-context-state sc))
	   (kana? (skk-context-kana-mode sc))
	   (str ""))
      (if kana?
	  (set! str "\tҤ餬\n")
	  (set! str "\t\n"))
      (if (= state 'skk-state-latin)
	  (set! str "S\tľ\n"))
      (if (= state 'skk-state-wide-latin)
	  (set! str "\tѱѿ\n"))
      (set! str (string-append "branch\t" str
			       "leaf\t\tҤ餬\tҤ餬ʥ⡼\tprop_skk_hiragana\n"
			       "leaf\t\t\tʥ⡼\tprop_skk_katakana\n"
			       "leaf\tS\tľ\tľϥ⡼\tprop_skk_latin\n"
			       "leaf\t\tѱѿ\tѱѿ⡼\tprop_skk_wide_latin\n"))
      (im-update-prop-list id str)
      )))

(register-im
 'skk
 "ja"
 "EUC-JP"
 nil
 skk-init-handler
 nil
 skk-mode-handler
 skk-press-key-handler
 skk-release-key-handler
 skk-reset-handler
 skk-get-candidate-handler
 skk-set-candidate-index-handler
 skk-prop-handler
 )
