;
(define global-case-insensitive-control-key? #f)
;
(define shift-key-mask
  (lambda (state)
    (= (bit-and state 1) 1)))
(define control-key-mask
  (lambda (state)
    (= (bit-and state 2) 2)))
(define alt-key-mask
  (lambda (state)
    (= (bit-and state 4) 4)))
(define meta-key-mask
  (lambda (state)
    (= (bit-and state 8) 8)))

(define modifier-key-mask
  (lambda (state)
    (> state 0)))

(define modifier-key
  (lambda (key key-state)
    (or
     (= key 'Shift_key)
     (= key 'Control_key)
     (= key 'Alt_key)
     (= key 'Meta_key))))

(define key-state-alist
  '((Shift_key   . 1)
    (Control_key . 2)
    (Alt_key     . 4)
    (Meta_key    . 8)))

(define emacs-like-prefix-alist
  '(("S" . Shift_key)
    ("C" . Control_key)
    ("A" . Alt_key)
    ("M" . Meta_key)
    ("I" . IgnoreCase)
    ("J" . IgnoreShift)))

(define tag-prefix-alist
  '(("Shift"       . Shift_key)
    ("Control"     . Control_key)
    ("Alt"         . Alt_key)
    ("Meta"        . Meta_key)
    ("IgnoreCase"  . IgnoreCase)
    ("IgnoreShift" . IgnoreShift)))

;;
(define backspace-key
  (lambda (key key-state)
    (= key 'backspace)))
(define left-key
  (lambda (key key-state)
    (= key 'left)))
(define right-key
  (lambda (key key-state)
    (= key 'right)))

(define switch-im-key
  (lambda (key state)
    (or
     (and (control-key-mask state)
	  (= key 'Shift_key))
     (and (shift-key-mask state)
	  (= key 'Control_key)))))

;;
(define intern-key-prefix
  (lambda (symbol-str alist)
    (let ((pair (assoc symbol-str alist)))
      (and pair
	   (cdr pair)))))

(define parse-tag-prefix-symbol
  (lambda (parsed char-list)
    (let ((prefix (if (string-equal? parsed "")
		      #f
		      (intern-key-prefix parsed tag-prefix-alist))))
      (if (not (null? char-list))
	  (let* ((head (car char-list))
		 (head-char (string->charcode head))
		 (rest (cdr char-list)))
	    (if (or (alphabet-char? head-char)
		    (numeral? head-char)
		    (string-equal? head "_"))
		(parse-tag-prefix-symbol (string-append parsed head) rest)
		(cons prefix char-list)))
	  (cons prefix ())))))

(define parse-tag-prefix
  (lambda (str)
    (if (not (string-equal? str ""))
	(let* ((char-list (reverse (string-to-list str)))
	       (head (car char-list)))
	  (if (string-equal? head "<")
	      (let* ((parsed (parse-tag-prefix-symbol "" (cdr char-list)))
		     (prefix (car parsed))
		     (rest (cdr parsed)))
		(if (and (not (null? rest))
			 (string-equal? (car rest) ">"))
		    (cons prefix
			  (if (null? (cdr rest))
			      ""
			      (apply string-append (cdr rest))))
		    (cons #f str)))
	      (cons #f str)))
	(cons #f str))))

(define parse-emacs-like-prefix
  (lambda (str)
    (let* ((char-list (reverse (string-to-list str)))
	   (prefix-str (and (<= 2 (length char-list))
			(string-equal? (nth 1 char-list) "-")
			(car char-list)))
	   (prefix (intern-key-prefix prefix-str emacs-like-prefix-alist))
	   (rest (if prefix
		     (apply string-append (cddr char-list))
		     str)))
      (cons prefix rest))))

(define parse-key-prefix
  (lambda (str)
    (let* ((parsed-as-emacs (parse-emacs-like-prefix str))
	   (prefix (car parsed-as-emacs)))
      (or (and prefix
	       parsed-as-emacs)
	  (parse-tag-prefix str)))))

(define parse-key-str
  (lambda (str translators key key-state)
    (let ((str-len (string-length str))
	  (rest ""))
      (cond
       ((= str-len 0)
	(list "" translators -1 key-state))
       ((= str-len 1)
	(list "" translators (string->charcode str) key-state))
       ((<= 2 str-len)
	(let* ((parsed (parse-key-prefix str))
	       (prefix (car parsed)))
	  (set! rest (cdr parsed))
	  (cond
	   ((modifier-key prefix 0)
	    (set! key-state
		  (bit-or key-state
			  (cdr (assq prefix key-state-alist)))))
	   ((= prefix 'IgnoreCase)
	    (set! translators
		  (cons (lambda (key key-state)
			  (let ((translated-key (to-lower-char key)))
			    (list translated-key key-state)))
			translators)))
	   ((= prefix 'IgnoreShift)
	    (set! translators
		  (cons (lambda (key key-state)
			  (let ((translated-key-state (bit-and key-state
							       (bit-not 1))))
			    (list key translated-key-state)))
			translators)))
	   (else
	    (set! key (intern str))	; keysym
	    (set! rest "")))
	  (if (string-equal? rest "")
	      (list rest translators key key-state)
	      (parse-key-str rest translators key key-state))))))))

(define apply-translators
  (lambda (translators key key-state)
    (if (null? translators)
	(list translators key key-state)
	(let* ((translator (car translators))
	       (rest-translators (cdr translators))
	       (translated (translator key key-state))
	       (translated-key (car translated))
	       (translated-state (cadr translated)))
	  (apply-translators
	   rest-translators
	   translated-key
	   translated-state)))))

(define make-single-key-predicate
  (lambda (source)
    (cond
     ((string? source)
      (let* ((key-str source)
	     (parsed (parse-key-str key-str () -1 0))
	     (translated (apply apply-translators (cdr parsed)))
	     (translators  (nth 1 parsed))
	     (target-key   (nth 1 translated))
	     (target-state (nth 2 translated)))
	(lambda (key key-state)
	  (let* ((translated (apply-translators translators key key-state))
		 (key       (nth 1 translated))
		 (key-state (nth 2 translated)))
	    (and (= key target-key)
		 (= key-state target-state))))))
     ((symbol? source)
      (let ((predicate-sym source))
	(lambda (key key-state)
	  ((symbol-value predicate-sym) key key-state))))
     (else
      (let ((maybe-predicate source))
	maybe-predicate)))))

(define make-key-predicate
  (lambda (sources)
    (cond
     ((list? sources)
      (let ((predicates (mapcar make-single-key-predicate sources)))
	(lambda (key key-state)
	  (apply proc-or
		 (mapcar (lambda (predicate)
			   (apply predicate (list key key-state)))
			 predicates)))))
     (else
      (let ((source sources))
	(make-single-key-predicate source))))))

(define modify-key-strs-implicitly
  (lambda (key-strs)
    (cond
     ((list? key-strs)
      (mapcar modify-key-strs-implicitly key-strs))
     ((string? key-strs)
      (let* ((key-str key-strs)
	     (modified-key-str (string-append
				"<IgnoreShift>"
				(if global-case-insensitive-control-key?
				    "<IgnoreCase>"
				    "")
				key-str)))
	modified-key-str))
     (else
      (let ((maybe-predicate key-strs))
	maybe-predicate)))))

(define define-key-internal
  (lambda (key-predicate-sym key-strs)
    (let* ((modified-key-strs (modify-key-strs-implicitly key-strs))
	   (predicate (make-key-predicate modified-key-strs))
	   (toplevel-env ()))
      (eval (list 'define key-predicate-sym predicate)
	    toplevel-env))))

;(define set-key-binding
;  (lambda (im state flavor)
;    nil
;    ))

;(define regist-key-binding 
;  (lambda (bind-name bind-table)
;    nil
;))
