// XIM Server currently for Japanese
// initialize many modules
//
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <ctype.h>
#include <stdlib.h>
#include <pwd.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include "xim.h"
#include "xdispatch.h"
#include "ximserver.h"

Display *XimServer::gDpy;
std::map<Window, XimServer *> XimServer::gServerMap;

// Configuration
int g_option_mask;
int scr_width,scr_height;
int host_byte_order;

int lib_uim_fd;
static char *result;

#define VERSION_NAME "uim-xim under the way! Version "PACKAGE_VERSION"\n"
char *version_name=VERSION_NAME;
char *usage=
"--help , --version :Show usage or version\n"
"--trace :trace-connection\n"
"--trace-xim :trace-xim-message\n";


static Atom atom_locales;
static Atom atom_transport;
Atom xim_servers;

// fd dispatch
#define READ_OK 1
#define WRITE_OK 2

struct fd_watch_struct {
    int mask;
    void (*fn)(int,int);
};
static std::map<int,fd_watch_struct> fd_watch_stat;
static std::map<unsigned int, WindowIf *> window_watch_stat;


bool
pretrans_register()
{
    xim_servers = XInternAtom(XimServer::gDpy, "XIM_SERVERS", 0);
    atom_locales = XInternAtom(XimServer::gDpy, "LOCALES", 0);
    atom_transport = XInternAtom(XimServer::gDpy, "TRANSPORT", 0);
    XFlush(XimServer::gDpy);
    scr_width = DisplayWidth(XimServer::gDpy, 0);
    scr_height = DisplayHeight(XimServer::gDpy, 0);
    return true;
}

WindowIf::~WindowIf()
{
}

void
WindowIf::resize(Window, int, int)
{
    // do nothing
}

static void remove_current_fd_watch(int fd)
{
    std::map<int,fd_watch_struct>::iterator i;
    i = fd_watch_stat.find(fd);
    if (i == fd_watch_stat.end()) {
        return ;
    }
    fd_watch_stat.erase(i);
}

static void add_fd_watch(int fd, int mask, void (*fn)(int, int))
{
    remove_current_fd_watch(fd);

    fd_watch_struct s;
    s.mask = mask;
    s.fn = fn;
    std::pair<int,fd_watch_struct> p(fd, s);
    fd_watch_stat.insert(p);
}

void main_loop()
{
    fd_set rfds, wfds;
    while (1) {
	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	std::map<int,fd_watch_struct>::iterator it;
	int  fd_max = 0;
	for (it = fd_watch_stat.begin(); it != fd_watch_stat.end(); it++) {
	    int fd = it->first;
	    if (it->second.mask & READ_OK) {
		FD_SET(fd, &rfds);
	    }
	    if (it->second.mask & WRITE_OK) {
		FD_SET(fd, &wfds);
	    }
	    if (fd_max < fd) {
		fd_max = fd;
	    }
	}
	select(fd_max + 1, &rfds, &wfds, NULL, NULL);
	for (it = fd_watch_stat.begin(); it != fd_watch_stat.end(); it++) {
	    int fd = it->first;
	    if (FD_ISSET(fd, &rfds)) {
		it->second.fn(fd, READ_OK);
	    }
	    if (FD_ISSET(fd, &wfds)) {
		it->second.fn(fd, WRITE_OK);
	    }
	}
    }
}

void
add_window_watch(Window id, WindowIf *w, int mask)
{
    std::pair<unsigned int,WindowIf *> p(id, w);
    window_watch_stat.insert(p);

    //Event mash has same value defined in X,
    //but do not depend on.
    int emask = 0;
    if (mask & EXPOSE_MASK) {
	emask |= ExposureMask;
    }
    if (mask & STRUCTURE_NOTIFY_MASK) {
	emask |= StructureNotifyMask;
    }
    XSelectInput(XimServer::gDpy, id, emask);
}

void
remove_window_watch(Window id)
{
    std::map<unsigned int,WindowIf *>::iterator i;
    i = window_watch_stat.find(id);
    if (i != window_watch_stat.end()) {
	window_watch_stat.erase(i);
    }
}

WindowIf *
findWindowIf(Window w)
{
    std::map<unsigned int,WindowIf *>::iterator i;
    i = window_watch_stat.find(w);
    if (i == window_watch_stat.end()) {
	return NULL;
    }
    return (*i).second;
}

static int
X_ErrorHandler(Display *d, XErrorEvent *e)
{
    if (g_option_mask & OPT_TRACE) {
	printf("X error occured.\n");
    }
    return 0;
}

static int
X_IOErrorHandler(Display *d)
{
    return 0;
}


static void
sendSelectionNotify(XEvent *ev, char *buf, int len)
{
    XEvent e;
    e.type = SelectionNotify;
    e.xselection.requestor = ev->xselectionrequest.requestor;
    e.xselection.selection = ev->xselectionrequest.selection;
    e.xselection.target = ev->xselectionrequest.target;
    e.xselection.time = ev->xselectionrequest.time;
    e.xselection.property = ev->xselectionrequest.property;
    XChangeProperty(XimServer::gDpy, e.xselection.requestor,
                    e.xselection.property,
                    e.xselection.target,
                    8,PropModeReplace,
                    (unsigned char *)buf, len);
    XSendEvent(XimServer::gDpy, e.xselection.requestor, 0, 0, &e);
    XFlush(XimServer::gDpy);
}

void
notifyLocale(XEvent *ev)
{
    char buf[32];
    XimServer *xs = XimServer::findServer(ev->xselectionrequest.owner);
    strcpy(buf,"@locale=ja_JP");
    sendSelectionNotify(ev, buf, strlen(buf)+1);
    if (g_option_mask & OPT_TRACE) {
	printf("selection notify request for locale.\n");
    }
}

void
notifyTransport(XEvent *ev)
{
    sendSelectionNotify(ev, "@transport=X/", 13+1);
    if (g_option_mask & OPT_TRACE) {
	printf("selection notify request for transport.\n");
    }
}

void
ProcXEvent(XEvent *e)
{
    Atom p;
    switch (e->type) {
    case SelectionRequest:
    {
	p = e->xselectionrequest.property;
	if (p == atom_locales) {
	    notifyLocale(e);
	} else if(p == atom_transport) {
	    notifyTransport(e);
	} else {
	    printf("property %s?\n",
		   XGetAtomName(XimServer::gDpy,e->xselection.property));
	    break;
	}
    }
    break;
    case Expose:
    {
	if (e->xexpose.count == 0) {
	    WindowIf *i = findWindowIf(e->xexpose.window);
	    if (i) {
		i->expose(e->xexpose.window);
	    }
	}
    }
    break;
    case ConfigureNotify:
    {
	WindowIf *i = findWindowIf(e->xconfigure.window);
	if (i) {
	    i->resize(e->xconfigure.window,
		      e->xconfigure.x, e->xconfigure.y);
	}
    }
    break;
    case DestroyNotify:
    {
	WindowIf *i = findWindowIf(e->xdestroywindow.window);
	if (i) {
	    i->destroy(e->xdestroywindow.window);
	}
	remove_window_watch(e->xdestroywindow.window);
    }
    break;
    case ClientMessage:
	procXClientMessage(&e->xclient);
    break;
    default:;
	//printf("unknown type of X event. %d\n",e->type);
    }
}

static void
xEventRead(int fd, int ev)
{
    XFlush(XimServer::gDpy);

    XEvent e;
    while (XPending(XimServer::gDpy)) {
	XNextEvent(XimServer::gDpy, &e);
	ProcXEvent(&e);
    }
}

static int
pretrans_setup()
{
    XSetErrorHandler(X_ErrorHandler);
    XSetIOErrorHandler(X_IOErrorHandler);
    int fd = XConnectionNumber(XimServer::gDpy);

    add_fd_watch(fd, READ_OK, xEventRead);
    return fd;
}

static void
print_version()
{
    printf(version_name);
}

static void
print_usage()
{
    print_version();
    printf(usage);
    exit(0);
}


static void
parse_args(int argc, char **argv)
{
    int i;
    for (i = 1; i < argc; i++) {
	if (!strncmp(argv[i], "--", 2)) {
	    char *opt;
	    opt = &argv[i][2];
	    if (!strcmp(opt, "version")) {
		print_version();
		exit(0);
	    } else if (!strcmp(opt, "help")) {
		print_usage();
	    } else if(!strcmp(opt, "trace")) {
		g_option_mask |= OPT_TRACE;
	    } else if(!strcmp(opt, "trace-xim")) {
		g_option_mask |= OPT_TRACE_XIM;
	    }
	}
    }
}

static void
get_runtime_env()
{
    int i = 1;
    char *v = (char *)&i;
    if (*v == 1) {
	host_byte_order = LSB_FIRST;
    } else {
	host_byte_order = MSB_FIRST;
    }
}

static void
helper_str_parse(char *str)
{
    //currently do nothing.
}

static void
helper_read_cb(int fd, int ev)
{
    uim_helper_read_proc(fd);
    char *tmp;
    while ((tmp = uim_helper_get_message())) {
	helper_str_parse(tmp);
	free(tmp);
    }
}

static void
helper_disconnect_cb(void)
{
    remove_current_fd_watch(lib_uim_fd);
    lib_uim_fd = -1;
}

int
main(int argc, char **argv)
{
    int res;
    printf("UIM-XIM bridge. Now only supports Japanese.\n");

    setlocale(LC_ALL, "");
    get_runtime_env();
    parse_args(argc, argv);

    lib_uim_fd = uim_helper_init_client_fd(helper_disconnect_cb);
    if (lib_uim_fd >= 0) {
	add_fd_watch(lib_uim_fd, READ_OK, helper_read_cb);
    }

    res = uim_init();
    if (res) {
	printf("Failed to init uim\n");
	return 1;
    }
    XimServer::gDpy = XOpenDisplay(NULL);
    if (!XimServer::gDpy) {
        printf("failed to open display!\n");
        return 1;
    }
    if (!pretrans_register()) {
	printf("pretrans_register failed\n");
	return 1;
    }
    uim_context uc = uim_create_context(NULL, "EUC-JP", NULL,
					NULL, NULL);
    int nr = uim_get_nr_im(uc);
    for (int i = 0; i < nr; i++) {
	char *name = (char *)uim_get_im_name(uc, i);
	char *lang = (char *)uim_get_im_language(uc, i);
	Locale *lc = getLocale(lang);
	if (!lc) {
	    continue;
	}
	XimServer *xs = new XimServer(lc, name, lang);
	bool res = xs->setupConnection();
	if (!res) {
	    delete xs;
	}
	printf("XMODIFIERS=@im=uim-%s ,(%s) registered\n", name, lang);
    }
    
    connection_setup();
    if (pretrans_setup() == -1) {
	return 0;
    }
    init_convdisp();

    main_loop();
    return 0;
}

int
pad4(int x)
{
    return (4-(x%4))%4;
}

void
hex_dump(unsigned char *buf, int len)
{
    int i, j;
    unsigned char c[16];
    for (i = 0 ; i < len ; i += 16) {
	printf("%8.8x ", i);
	for (j = 0 ; j < 16 ; j++) {
	    if (i + j < len) {
		c[j] = buf[i+j];
		if (j == 7) {
		    printf("%2.2x-", c[j]);
		} else {
		    printf("%2.2x ", c[j]);
		}
	    } else {
		c[j] = 0;
		printf("-- ");
	    }
	}
	printf(" ");
	for (j = 0; j < 16; j++) {
	    if (isprint(c[j])) {
		printf("%c", c[j]);
	    } else {
		printf(".");
	    }
	}
	printf("\n");
    }
}
/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 * End:
 */
