;;; canna.scm: Canna for uim.
;;;
;;; Copyright (c) 2003,2004 Anthy Project http://anthy.sourceforge.jp/
;;; Copyright (c) 2003,2004 uim Project http://uim.freedesktop.org/
;;; Copyright (c) 2003,2004 Masahito Omote <omote@utyuuzin.net>
;;;
;;; You can use, modify, distribute under the term of GPL or BSD Licence.
;;;;
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU Library General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
;;;
;;;;
;;;
;;; Copyright (c) 2003,2004 Anthy Project http://anthy.sourceforge.jp/
;;; Copyright (c) 2003,2004 uim Project http://uim.freedesktop.org/
;;; Copyright (c) 2003,2004 Masahito Omote <omote@utyuuzin.net>
;;;
;;; All rights reserved.
;;;
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above copyright
;;;    notice, this list of conditions and the following disclaimer in the
;;;    documentation and/or other materials provided with the distribution.
;;; 3. Neither the name of authors nor the names of its contributors
;;;    may be used to endorse or promote products derived from this software
;;;    without specific prior written permission.
;;;
;;; THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
;;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
;;; FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
;;; OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;;; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
;;; LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
;;; OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
;;; SUCH DAMAGE.
;;;
;;;;

(require "japanese.scm")
(require "generic-key.scm")
(require "generic.scm")

; variables
(define canna-init-lib-ok? nil)
(define canna-use-candidate-window? #t)
(define canna-candidate-op-count 1)
; TODO: support cannaserver on other host
(define canna-server-name nil)
;(define canna-server-name "localhost")
;(define canna-server-name "127.0.0.1")
;;; Key definition
(define (canna-on-key? key key-state)
  (and
   (= key 92)
   (control-key-mask key-state)))
(define (canna-off-key? key key-state)
  (and
   (= key 92)
   (control-key-mask key-state)))
(define (canna-latin-key key key-state)
  (and
   (= key 92)
   (control-key-mask key-state)))
;(define (canna-latin-key key key-state)
;  (or
;   (and
;    (= (to-lower-char key) 108)
;    (not (modifier-key-mask key-state)))
;   (generic-on-key key key-state)))
(define (canna-wide-latin-key key key-state)
  (and (= (to-lower-char key) 108)
       (shift-key-mask key-state)))
(define (canna-kana-toggle-key key key-state)
  (and
   (= (to-lower-char key) 113)
   (not (modifier-key-mask key-state))))
(define (canna-cancel-key key key-state)
    (generic-cancel-key key key-state))
(define (canna-backspace-key key key-state)
  (generic-backspace-key key key-state))
(define (canna-delete-key key key-state)
  (generic-delete-key key key-state))
(define (canna-go-left-key key key-state)
  (generic-go-left-key key key-state))
(define (canna-go-right-key key key-state)
  (generic-go-right-key key key-state))
(define (canna-commit-key key key-state)
  (generic-commit-key key key-state))
(define (canna-begin-conv-key? key key-state)
  (or (= (to-lower-char key) 32)
      (generic-on-key key key-state)))
(define (canna-commit-key? key key-state)
  (generic-commit-key key key-state))
(define (canna-extend-segment-key? key key-state)
  (or (and (control-key-mask key-state)
	   (= (to-lower-char key) 111))
      (and (shift-key-mask key-state)
	   (= key 'right))))
(define (canna-shrink-segment-key? key key-state)
  (or (and (control-key-mask key-state)
	   (= (to-lower-char key) 105))
      (and (shift-key-mask key-state)
	   (= key 'left))))
(define (canna-next-candidate-key? key key-state)
  (generic-next-candidate-key key key-state))
(define (canna-prev-candidate-key? key key-state)
  (generic-prev-candidate-key key key-state))
(define (canna-cancel-key? key key-state)
  (generic-cancel-key key key-state))
(define (canna-delete-key? key key-state)
  (generic-delete-key key key-state))
(define (canna-go-left-key? key key-state)
    (generic-go-left-key key key-state))
(define (canna-go-right-key? key key-state)
    (generic-go-right-key key key-state))
(define (canna-next-segment-key? key key-state)
    (generic-go-right-key key key-state))
(define (canna-prev-segment-key? key key-state)
    (generic-go-left-key key key-state))

;;; access
(define (canna-context-id cc)
  (car (nthcdr 0 cc)))
(define (canna-context-set-id! cc id)
  (set-car! (nthcdr 0 cc) id))

(define (canna-context-on cc)
  (car (nthcdr 0 cc)))
(define (canna-context-set-on! cc s)
  (set-car! (nthcdr 0 cc) s))

(define (canna-context-state cc)
  (car (nthcdr 1 cc)))
(define (canna-context-set-state! cc st)
  (set-car! (nthcdr 1 cc) st))

(define (canna-context-cc-id cc)
  (car (nthcdr 2 cc)))
(define (canna-context-set-cc-id! cc id)
  (set-car! (nthcdr 2 cc) id))

(define (canna-context-left-string cc)
  (car (nthcdr 3 cc)))
(define (canna-context-set-left-string! cc str)
  (set-car! (nthcdr 3 cc) str))

(define (canna-context-right-string cc)
  (car (nthcdr 4 cc)))
(define (canna-context-set-right-string! cc str)
  (set-car! (nthcdr 4 cc) str))

(define (canna-context-rkc cc)
  (car (nthcdr 5 cc)))
(define (canna-context-set-rkc! cc rkc)
  (set-car! (nthcdr 5 cc) rkc))

(define (canna-context-index-list cc)
  (car (nthcdr 6 cc)))
(define (canna-context-set-index-list! cc lst)
  (set-car! (nthcdr 6 cc) lst))

(define (canna-context-cur-seg cc)
  (car (nthcdr 7 cc)))
(define (canna-context-set-cur-seg! cc seg)
  (set-car! (nthcdr 7 cc) seg))

(define (canna-context-candidate-window cc)
  (car (nthcdr 8 cc)))
(define (canna-context-set-candidate-window! cc f)
  (set-car! (nthcdr 8 cc) f))

(define (canna-context-candidate-op-count cc)
  (car (nthcdr 9 cc)))
(define (canna-context-set-candidate-op-count! cc c)
  (set-car! (nthcdr 9 cc) c))

(define (canna-context-wide-latin cc)
  (car (nthcdr 10 cc)))
(define (canna-context-set-wide-latin! cc c)
  (set-car! (nthcdr 10 cc) c))

(define (canna-context-kana-mode cc)
  (car (nthcdr 11 cc)))
(define (canna-context-set-kana-mode! cc c)
  (set-car! (nthcdr 11 cc) c))

;;; on/off, state, canna-context-id, string, rkc, index-list, cur-seg, candidate-window, candidate-op-count, wide-latin, kana-mode
(define (canna-context-new)
  (let ((c '())
	(rkc (rk-context-new ja-rk-rule #t #f)))
    (set! c (copy-list '(() () () () () () () () () () () #f #t)))
    (canna-context-set-cc-id! c (if canna-init-lib-ok?
				    (canna-lib-alloc-context) ()))
    (canna-context-set-rkc! c rkc)
    (canna-flush c)
    (canna-context-set-on! c #f)
    c))

(define (canna-context-kana-toggle cc)
  (canna-context-set-kana-mode! cc (not (canna-context-kana-mode cc))))

(define (canna-make-string sl dir kana)
  (if sl
      (if dir
	  (string-append (canna-make-string (cdr sl) dir kana)
			 (if kana
			     (cdar sl)
			     (caar sl)))
	  (string-append (if kana
			     (cdar sl)
			     (caar sl))
			 (canna-make-string (cdr sl) dir kana)))
      ""))

(define (canna-make-left-string sl kana)
  (canna-make-string sl #t kana))

(define (canna-make-right-string sl kana)
  (canna-make-string sl #f kana))

(define (canna-mode-handler id mode)
  (let* ((c (find-context id))
	 (cc (context-date c)))
    (canna-flush cc)
    (cond
     ((= mode 0)
      (canna-context-set-on! cc #f)
      (canna-context-set-wide-latin! cc #f))
     ((= mode 1)
      (canna-context-set-on! cc #t)
      (canna-context-set-kana-mode! cc #f))
     ((= mode 2)
      (canna-context-set-on! cc #t)
      (canna-context-set-kana-mode! cc #t))
     ((= mode 3)
      (canna-context-set-on! cc #f)
      (canna-context-set-wide-latin! cc #t)))
    (canna-update-preedit cc id))
  ())

(define (canna-flush cc)
  (rk-flush (canna-context-rkc cc))
  (canna-context-set-left-string! cc '())
  (canna-context-set-right-string! cc '())
  (canna-context-set-on! cc #t)
  (canna-context-set-state! cc #f)
  (canna-context-set-index-list! cc ())
  (canna-context-set-candidate-window! cc #f)
  (canna-context-set-candidate-op-count! cc 0))

(define (canna-begin-input cc)
  (canna-context-set-on! cc #t)
  (rk-flush (canna-context-rkc cc))
  (canna-context-set-state! cc #f))

(define (canna-update-preedit cc id)
  (if (canna-context-on cc)
      (if (canna-context-state cc)
	  (canna-compose-state-preedit cc id)
	  (canna-input-state-preedit cc id))
      (begin
	(im-clear-preedit id)
	(im-update-preedit id))))

(define (canna-make-index-list-rec n)
  (if (> n 0)
      (cons 0
	    (canna-make-index-list-rec (- n 1)))
      '()))

(define (canna-make-index-list n old-lst)
  (if (< n (length old-lst))
      (truncate-list old-lst n 1)
      (append old-lst
	      (canna-make-index-list-rec (- n
					    (length old-lst))))))

(define (canna-append-string cc str)
  (and str
       (if (not (string? (car str)))
	   (begin
	     (canna-append-string cc (car str))
	     (canna-append-string cc (cdr str))
	     #f)
	   #t)
       (canna-context-set-left-string!
	cc (cons str (canna-context-left-string cc)))))

(define (canna-begin-conv cc)
  (let* ((cc-id (canna-context-cc-id cc))
	 (rkc (canna-context-rkc cc))
	 (kana (canna-context-kana-mode cc))
	 (last "")
	 (res))
    (set! res (rk-push-key-last! rkc))
    (if res
	(canna-append-string cc res))

    (canna-context-set-index-list!
     cc
     (canna-make-index-list
      (canna-lib-begin-conversion
       cc-id
       (string-append
	(canna-make-left-string (canna-context-left-string cc) #f)
	(canna-make-right-string (canna-context-right-string cc) #f))) nil))
    (canna-context-set-state! cc #t)
    (canna-context-set-cur-seg! cc 0)
    (rk-flush (canna-context-rkc cc))))


(define (canna-proc-input-state-no-preedit cc id key key-state)
  (let
      ((rkc (canna-context-rkc cc))
       (direct (ja-direct (charcode->string key))))
    (and
     (if (canna-wide-latin-key key key-state)
	 (begin
	   (canna-flush cc)
	   (canna-context-set-on! cc #f)
	   (canna-context-set-wide-latin! cc #t)
	   (canna-update-mode cc id)
	   (canna-update-prop-label cc id)
	   #f)
	 #t)
     (if (canna-latin-key key key-state)
	 (begin
	   (canna-flush cc)
	   (canna-context-set-on! cc #f)
	   (canna-context-set-wide-latin! cc #f)
	   (canna-update-mode cc id)
	   (canna-update-prop-label cc id)
	   #f)
	 #t)
     (if (canna-backspace-key key key-state)
	 (begin
	   (im-commit-raw id)
	   #f)
	 #t)
     (if (canna-delete-key key key-state)
	 (begin
	   (im-commit-raw id)
	   #f)
	 #t)
     (if (canna-kana-toggle-key key key-state)
	 (begin
	   (canna-context-kana-toggle cc)
	   (canna-update-mode cc id)
	   (canna-update-prop-label cc id)
	   #f)
	 #t)
     ;; modifiers (except shift) => ignore
     (if (and (modifier-key-mask key-state)
	      (not (shift-key-mask key-state)))
	 (begin
	   (im-commit-raw id)
	   #f)
	 #t)
     ;; direct key => commit
     (if direct
	 (begin
	   (im-commit id direct)
	   #f)
	 #t)
     (set! key-str (charcode->string (to-lower-char key)))
     (if (string-find (rk-expect rkc) key-str)
	 (let ((res))
	   (set! res (rk-push-key! rkc key-str))
	   (if res
	       (canna-append-string cc res))
	   #f)
	 #t)
     (im-commit-raw id))))

(define (canna-has-preedit? cc)
  (or
   (> (length (canna-context-left-string cc)) 0)
   (> (length (canna-context-right-string cc)) 0)
   (> (length (rk-pending (canna-context-rkc cc))) 0)))


(define (canna-proc-input-state-with-preedit cc id key key-state)
  (let ((rkc (canna-context-rkc cc))
	(kana (canna-context-kana-mode cc)))
    (and
       ;; begin conversion
     (if (and
	  (canna-begin-conv-key? key key-state)
	  canna-init-lib-ok?)
	 (begin
	   (canna-begin-conv cc)
	   #f)
	 #t)
     ;; backspace
     (if (canna-backspace-key key key-state)
	 (begin
	   (if (not (rk-backspace rkc))
	       (if (canna-context-left-string cc)
		   (canna-context-set-left-string!
		    cc
		    (cdr (canna-context-left-string cc)))))
	   #f)
	 #t)
     ;; delete
     (if (canna-delete-key key key-state)
	 (begin
	   (if (not (rk-delete rkc))
	       (if (canna-context-right-string cc)
		   (canna-context-set-right-string!
		    cc
		    (cdr (canna-context-right-string cc)))))
	   #f)
	 #t)
       ;; Ҥ餬ʥ⡼ɤǥʤꤹ
     (if (canna-kana-toggle-key key key-state)
	 (begin
	   (im-commit
	    id
	    (string-append
	     (canna-make-left-string (canna-context-left-string cc)
				     (not kana))
	     (canna-make-right-string (canna-context-right-string cc)
				      (not kana))))
	   (canna-flush cc)
	   #f)
	 #t)
       ;; cancel
     (if (canna-cancel-key? key key-state)
	 (begin
	   (canna-flush cc)
	   #f)
	 #t)
       ;; commit
     (if (canna-commit-key? key key-state)
	 (begin
	   (im-commit
	    id
	    (string-append
	     (canna-make-left-string (canna-context-left-string cc) kana)
	     (canna-make-right-string (canna-context-right-string cc) kana)))
	   (canna-flush cc)
	   #f)
	 #t)
       ;; left
     (if (canna-go-left-key? key key-state)
	 (begin
	   (if (canna-context-left-string cc)
	       (let
		   ((c (car (canna-context-left-string cc))))
		 (canna-context-set-left-string!
		  cc (cdr (canna-context-left-string cc)))
		 (canna-context-set-right-string!
		  cc
		  (cons c (canna-context-right-string cc)))))
	   #f)
	 #t)
       ;; right
     (if (canna-go-right-key? key key-state)
	 (begin
	   (if (canna-context-right-string cc)
	       (let
		   ((c (car (canna-context-right-string cc))))
		 (canna-context-set-right-string!
		  cc (cdr (canna-context-right-string cc)))
		 (canna-append-string cc c)))
	   #f)
	 #t)
     ;; modifiers (except shift) => ignore
     (if (and (modifier-key-mask key-state)
	      (not (shift-key-mask key-state)))
	 (begin
	   (im-commit-raw id)
	   #f)
	 #t)
     (set! key (to-lower-char key))
     (let ((res)
	   (key-str (charcode->string key)))
       (set! res (rk-push-key! rkc key-str))
       (if res
	   (canna-append-string cc res))))))

(define (canna-proc-input-state cc id key key-state)
  (if (canna-has-preedit? cc)
      (canna-proc-input-state-with-preedit cc id key key-state)
      (canna-proc-input-state-no-preedit cc id key key-state)))

(define (canna-pushback-preedit-segment-rec cc id idx nseg)
  (let ((cc-id (canna-context-cc-id cc)))
    (if (< idx nseg)
	(begin
	  (im-pushback-preedit
	   id
	   (if (= idx (canna-context-cur-seg cc))
	       (+ preedit-reverse preedit-cursor)
	       preedit-underline)
	   (canna-lib-get-nth-candidate
	      cc-id idx
	      (nth idx (canna-context-index-list cc))))
	    (canna-pushback-preedit-segment-rec cc id (+ idx 1) nseg)))))

(define (canna-compose-state-preedit cc id)
  (im-clear-preedit id)
  (canna-pushback-preedit-segment-rec
   cc id
   0 (length (canna-context-index-list cc)))
  (im-update-preedit id))

(define (canna-input-state-preedit cc id)
  (let ((rkc (canna-context-rkc cc))
	(kana (canna-context-kana-mode cc)))
    (im-clear-preedit id)
    (im-pushback-preedit
     id preedit-underline
     (canna-make-left-string (canna-context-left-string cc) kana))
    (im-pushback-preedit id preedit-underline
			 (rk-pending rkc))
    (if (canna-has-preedit? cc)
	(im-pushback-preedit id preedit-cursor ""))
    (im-pushback-preedit
     id preedit-underline
     (canna-make-right-string (canna-context-right-string cc) kana))
    (im-update-preedit id)))

(define (canna-get-commit-string cc idx nseg)
  (let ((cc-id (canna-context-cc-id cc)))
    (if (< idx nseg)
	(string-append
	 (canna-lib-get-nth-candidate
	  cc-id idx
	  (nth idx (canna-context-index-list cc)))
	 (canna-get-commit-string cc (+ idx 1) nseg))
	"")))

(define (canna-commit-string cc idx nseg)
  (let ((cc-id (canna-context-cc-id cc)))
    (if (< idx nseg)
	(begin
	  (canna-lib-commit-segment
	   cc-id idx (nth idx (canna-context-index-list cc)))
	  (canna-commit-string cc
			       (+ idx 1) nseg))
	nil)))

(define (canna-do-commit cc id)
    (canna-reset-candidate-window cc id)
    (im-commit id
	       (canna-get-commit-string
		cc 0
		(length (canna-context-index-list cc))))
    (canna-commit-string
     cc 0
     (length (canna-context-index-list cc)))
    (canna-flush cc))

(define (canna-init-handler id arg)
  (let ((c (find-context id)))
    (set-context-data! c (canna-context-new))
    (im-clear-mode-list id)
    (im-pushback-mode-list id "ľ")
    (im-pushback-mode-list id "Ҥ餬")
    (im-pushback-mode-list id "")
    (im-pushback-mode-list id "ѱѿ")
    (im-update-mode-list id)
    (im-update-mode id 0)
    ()))

(define (canna-release-handler id)
  (let* ((c (find-context id))
	 (cc (context-data c))
	 (cc-id (canna-context-cc-id cc)))
    (canna-lib-release-context cc-id)))

(define (canna-mode-handler id mode)
  (let* ((c (find-context id))
	 (cc (context-data c)))
    (canna-flush cc)
    (cond  ; `case' is not supported by uim
     ((= mode 0)  ; 'direct'
      (canna-context-set-on! cc #f)
      (canna-context-set-wide-latin! cc #f))
     ((= mode 1)  ; 'hiragana'
      (canna-context-set-on! cc #t)
      (canna-context-set-kana-mode! cc #f))
     ((= mode 2)  ; 'katakana'
      (canna-context-set-on! cc #t)
      (canna-context-set-kana-mode! cc #t))
     ((= mode 3)  ; 'wide-latin'
      (canna-context-set-on! cc #f)
      (canna-context-set-wide-latin! cc #t))) 
    (canna-update-preedit cc id))
  ())

(define (canna-move-segment cc dir)
  (let ((pos (+ (canna-context-cur-seg cc) dir))
	(nseg (length (canna-context-index-list cc))))
      (if (and
	   (> pos -1)
	   (< pos nseg))
	  (canna-context-set-cur-seg! cc pos))))

(define (canna-move-candidate cc id off)
  (let* ((seg (canna-context-cur-seg cc))
	 (n (nth seg (canna-context-index-list cc)))
	 (cc-id (canna-context-cc-id cc))
	 (max (canna-lib-get-nr-candidates cc-id seg)))
    (set! n (+ n off))
    (if (>= n max)
	(set! n 0))
    (if (< n 0)
	(set! n (- max 1)))
    (set-car! (nthcdr seg (canna-context-index-list cc)) n)
    (canna-context-set-candidate-op-count!
     cc
     (+ 1 (canna-context-candidate-op-count cc)))
    (if (and
	 (= (canna-context-candidate-op-count cc)
	    canna-candidate-op-count)
	 canna-use-candidate-window?)
	(begin
	  (canna-context-set-candidate-window! cc #t)
	  (im-begin-candidate
	   id
	   max 0)))
    (if (canna-context-candidate-window cc)
	(im-update-candidate id n))))

(define (canna-reset-candidate-window cc id)
  (if (canna-context-candidate-window cc)
      (begin
	(im-end-candidate id cc)
	(canna-context-set-candidate-window! cc #f)))
  (canna-context-set-candidate-op-count! cc 0))

(define (canna-resize-segment cc id cnt)
  (let
      ((cc-id (canna-context-cc-id cc)))
      (canna-reset-candidate-window cc id)
      (canna-lib-resize-segment
       cc-id (canna-context-cur-seg cc) cnt)
      (canna-context-set-index-list!
       cc
       (canna-make-index-list
	(canna-lib-get-nr-segments cc-id)
	(truncate-list
	 (canna-context-index-list cc)
	 (canna-context-cur-seg cc)))
      )))

(define (canna-proc-compose-state cc id key key-state)
  (let
      ((cc-id (canna-context-cc-id cc)))
    (and
     (if (canna-commit-key key key-state)
	 (begin
	   (canna-do-commit cc id)
	   #f)
	 #t)
     (if (canna-extend-segment-key? key key-state)
	 (begin
	   (canna-resize-segment cc id 1)
	   #f)
	 #t)
     (if (canna-shrink-segment-key? key key-state)
	 (begin
	   (canna-resize-segment cc id -1)
	   #f)
	 #t)
     (if (canna-next-segment-key? key key-state)
	 (begin
	   (canna-move-segment cc 1)
	   (canna-reset-candidate-window cc id)
	   #f)
	 #t)
     (if (canna-prev-segment-key? key key-state)
	 (begin
	   (canna-move-segment cc -1)
	   (canna-reset-candidate-window cc id)
	   #f)
	 #t)
     (if (canna-backspace-key key key-state)
	 (begin
	   (canna-context-set-state! cc #f)
	   (canna-reset-candidate-window cc id)
	   #f)
	 #t)
     (if (canna-next-candidate-key? key key-state)
	 (begin
	   (canna-move-candidate cc id 1)
	   #f)
	 #t)
     (if (canna-prev-candidate-key? key key-state)
	 (begin
	   (canna-move-candidate cc id -1)
	   #f)
	 #t)
     (if (canna-cancel-key? key key-state)
	 (begin
	   (canna-context-set-state! cc #f)
	   (canna-reset-candidate-window cc id)
	   (canna-lib-reset-conversion cc-id)
	   #f)
	 #t)
     (if (and (modifier-key-mask key-state)
	      (not (shift-key-mask key-state)))
	 #f
	 #t)
     (begin
       (canna-do-commit cc id)
       (canna-proc-input-state cc id key key-state)))
    ()))

(define (canna-proc-wide-latin c key key-state)
  (let* ((w (ja-wide (charcode->string key)))
	 (id (context-id c))
	 (cc (context-data c)))
    (and
     (if (canna-on-key? key key-state)
	 (begin
	   (canna-flush cc)
	   (canna-update-mode cc id)
	   (canna-update-prop-label cc id)
	   #f)
	 #t)
     (if (and (modifier-key-mask key-state)
	      (not (shift-key-mask key-state)))
	 (begin
	   (im-commit-raw id)
	   #f)
	 #t)
     (if w
	 (im-commit id w)
	 (im-commit-raw id)))
    ()))

(define (canna-press-key-handler id key key-state)
  (let* ((c (find-context id))
	 (cc (context-data c)))
    (if (canna-context-on cc)
	(if (canna-context-state cc)
	    (canna-proc-compose-state cc id key key-state)
	    (canna-proc-input-state cc id key key-state))
	(if (canna-context-wide-latin cc)
	    (canna-proc-wide-latin c key key-state)
	    (canna-proc-raw-state c key key-state)))
    (canna-update-preedit cc id)))

;;;
(define (canna-release-key-handler id key key-state)
  ())
;;;
(define (canna-reset-handler id)
  ())

;;;
(define (canna-get-candidate-handler id idx)
  (let* ((c (find-context id))
	  (cc (context-data c))
	  (cc-id (canna-context-cc-id cc)))
    (canna-lib-get-nth-candidate
     cc-id (canna-context-cur-seg cc)
     idx)))

(define (canna-set-candidate-index-handler id idx)
  (let* ((c (find-context id))
	 (cc (context-data c))
	 (seg (canna-context-cur-seg cc))
	 (cc-id (canna-context-cc-id cc)))
    (set-car! (nthcdr seg (canna-context-index-list cc)) idx)
    (canna-update-preedit cc id)))

;;;

(define (canna-prop-handler id message)
  (let* ((c (find-context id))
	 (cc (context-data c)))
    (canna-flush cc)
    (canna-update-preedit cc id)
    (if (string-equal? message
		       "prop_canna_hiragana")
	(begin
	  (canna-context-set-on! cc #t)
	  (canna-context-set-kana-mode! cc #t)))
    (if (string-equal? message
		       "prop_canna_katakana")
	(begin
	  (canna-context-set-on! cc #t)
	  (canna-context-set-kana-mode! cc #f)))
    (if (string-equal? message
		       "prop_canna_direct")
	(begin
	  (canna-context-set-on! cc #f)
	  (canna-context-set-wide-latin! cc #f)))
    (if (string-equal? message
		       "prop_canna_zenkaku")
	(begin
	  (canna-context-set-on! cc #f)
	  (canna-context-set-wide-latin! cc #t)))
    (canna-update-mode cc id)
    (canna-update-prop-label cc id)))
;;;

(define (canna-proc-raw-state c key key-state)
    (let ((id (context-id c))
	  (cc (context-data c)))
      (if (canna-on-key? key key-state)
	  (begin
	    (canna-begin-input cc)
	    (canna-update-mode cc id)
	    (canna-update-prop-label cc id))
	  (im-commit-raw (context-id c))
	  (canna-update-preedit)
)))

(define (canna-update-prop-label cc id)
  (let ((str ""))
    (set! str
	  (if (canna-context-on cc)
	      (if (canna-context-kana-mode cc)
		  "\t\n"
		  "\tҤ餬\n")
	      (if (canna-context-wide-latin cc)
		  "\tѱѿ\n"
		  "a\tľ\n")))
    (set! str (string-append str "\tϥ⡼\n"))
    (im-update-prop-label id str)))

(define (canna-update-mode cc id)
  (if (canna-context-on cc)
      (if (canna-context-kana-mode cc)
	  (im-update-mode id 2)
	  (im-update-mode id 1))
      (if (canna-context-wide-latin cc)
	  (im-update-mode id 3)
	  (im-update-mode id 0)))
  (canna-update-prop-label cc id))

(define (canna-update-prop-list id)
  (let* ((c (find-context id))
	 (cc (context-data c))
	 (str "branch\t"))
    (set! str
	  (string-append str
			 (if (canna-context-on cc)
			     "\tҤ餬\n"
			     (if (canna-context-wide-latin cc)
				 "\tѱѿ\n"
				 "a\tľ\n"))))
    (set! str
	  (string-append
	   str
	   "leaf\t\tҤ餬\tҤ餬ʤǤ\tprop_canna_hiragana\n"
	   "leaf\t\t\tʤϤǤޤ\tprop_canna_katakana\n"
	   "leaf\tA\tľ\t쥯ȤǤ\tprop_canna_direct\n"
	   "leaf\t\tѱѿ\tѱѿ⡼\tprop_canna_zenkaku\n"
	   "branch\t\t\nleaf\t\t޻\tdesc\tprop_canna_roma\n"
	   "leaf\t\t\t\tdescription\tprop_canna_kana\n"))
    (im-update-prop-list id str)))

(define canna-prop-list
  '(
    ("branch" "Ͼ")
    ("leaf" "" "Ҥ餬" "Ҥ餬ʤϤǤޤ" "prop_canna_hiragana")
    ("leaf" "" "" "" "prop_canna_katakana")
    ("leaf" "" "ѱѿ" "ѱѿ⡼" "prop_canna_zenkaku")
    ("leaf" "a" "ľ" "ե٥åȤϤǤޤ" "prop_canna_direct")
    ("branch" "ϥ⡼")
    ("leaf" "" "޻" "޻" "prop_canna_roma")
    ("leaf" "" "" "" "prop_canna_kana")))

(if (and
     (symbol-bound? 'canna-lib-init)
     (canna-lib-init canna-server-name))
    (set! canna-init-lib-ok? #t))

(if (and
     (symbol-bound? 'canna-lib-init)
     (= canna-init-lib-ok? #t))
    (register-im
     'canna
     "ja"
    "EUC-JP"
     nil
     canna-init-handler
     canna-release-handler
     canna-mode-handler
     canna-press-key-handler
     canna-release-key-handler
     canna-reset-handler
     canna-get-candidate-handler
     canna-set-candidate-index-handler
     canna-prop-handler))
